﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

// 同一プロセスでサービスを処理
#define NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC       0
// 別プロセスでサービスを処理
#define NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC      1

// どちらのモードで動作させるかのコンフィグ値
#if defined(NN_BUILD_CONFIG_OS_WIN)
#define NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS           NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC
// 自動的に終了処理を行えるようにする
#define NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION
#else
#define NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS           NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
// 終了処理は省略する
#undef NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION
#endif

#if !defined(NN_SDK_BUILD_RELEASE)
// 非リリース版ではRecoveryはフル権限とする(主にデバッグ/テスト用)
#define NN_DETAIL_PCTL_CONFIG_FULL_CAPABILITY_FOR_RECOVERY
#endif

namespace nn { namespace pctl { namespace detail { namespace ipc {

// IPCを行うためのセッション数の設定
static const size_t SessionCountPerProcess           = 4; // FactoryとServiceの最大2つ*2を同時利用

static const size_t ProcessCountMaxForGeneral        = 6; // メインアプリ＋Scoop-Postなど
static const size_t ProcessCountMaxForSystem         = 8; // 複数SA・LAが動く可能性あり
static const size_t ProcessCountMaxForAuthentication = 1; // 認証機能のみ
static const size_t ProcessCountMaxForRecovery       = 1; // 修理ツールのみ
static const size_t ProcessCountMax                  = 16;// (上記5つの合計値)
NN_STATIC_ASSERT(ProcessCountMax ==
    ProcessCountMaxForGeneral + ProcessCountMaxForSystem +
    ProcessCountMaxForAuthentication + ProcessCountMaxForRecovery
    );
static const size_t SessionCountMaxForGeneral        = SessionCountPerProcess * ProcessCountMaxForGeneral;
static const size_t SessionCountMaxForSystem         = SessionCountPerProcess * ProcessCountMaxForSystem;
static const size_t SessionCountMaxForAuthentication = SessionCountPerProcess * ProcessCountMaxForAuthentication;
static const size_t SessionCountMaxForRecovery       = SessionCountPerProcess * ProcessCountMaxForRecovery;
static const size_t SessionCountMax                  = SessionCountPerProcess * ProcessCountMax;
NN_STATIC_ASSERT(SessionCountMax ==
    SessionCountMaxForGeneral + SessionCountMaxForSystem +
    SessionCountMaxForAuthentication + SessionCountMaxForRecovery
    );

// サービス(ポート)をインデックス値に表した定数
enum ServiceIndex
{
    ServiceIndex_General = 0,
    ServiceIndex_System,
    ServiceIndex_Authentication,
    ServiceIndex_Recovery,
    ServiceIndex_Count
};

// サービス(ポート)の文字列表記
// (const char* 定義だと ServiceNameList の初期化が static initializer になる)
static const char ServiceNameForGeneral[] = "pctl";   // 一般権限(SNS投稿関連機能向け含む)
static const char ServiceNameForSystem[] = "pctl:s";  // システム権限
static const char ServiceNameForAuthentication[] = "pctl:a";  // 認証機能向け権限(全権限持たせる)
static const char ServiceNameForRecovery[] = "pctl:r";  // 修理機能向け権限

// 利用可能な機能を示す値
enum Capability
{
    Capability_None          = 0x0000, // なし
    Capability_Application   = 0x0001, // 一般アプリケーション (pctl)
    Capability_SnsCheck      = 0x0002, // SNS投稿機能制限関連 (pctl, pctl:s, pctl:a)
    Capability_Management    = 0x0004, // 設定変更などの管理機能 (pctl:s, pctl:a)
    Capability_Watcher       = 0x0008, // 見守り機能 (pctl:s, pctl:a)
    Capability_PinCode       = 0x0010, // 解除コード管理機能 (pctl:a)
    Capability_MasterKey     = 0x0020, // マスターキー管理機能 (pctl:a)
    Capability_Recovery      = 0x0040, // 修理向け機能 (pctl:r)
    Capability_RatingCheck   = 0x0080, // レーティング制限確認機能 (pctl:s, pctl:a)
    Capability_ReadStatus    = 0x0100, // 各種設定・状態の読み取り機能 (pctl ← 特にリスクは無いため一般向けで利用可とする)
    Capability_StereoVision  = 0x0200, // 立体視機能制限の設定 (pctl, pctl:s, pctl:a)
    Capability_System        = 0x8000, // システム(起動終了を明示的に記録しないようにする) (pctl:s, pctl:a, pctl:r)
    Capability_Full          = 0x7FFF, // フル権限 (Generic向け; アプリケーション扱いにするため「System」を除く)
    Capability_All           = 0xFFFF  // 全権限
};
// サービス(ポート)が持つCapability
static const int ServiceCapabilityForGeneral = Capability_Application |
    Capability_SnsCheck | Capability_ReadStatus | Capability_StereoVision; // アプリ向けでも状態読み取りやSNSチェックを使えるようにする
static const int ServiceCapabilityForSystem = Capability_System | Capability_SnsCheck |
    Capability_Management | Capability_Watcher | Capability_RatingCheck | Capability_ReadStatus | Capability_StereoVision;
static const int ServiceCapabilityForAuthentication = Capability_System | Capability_SnsCheck |
    Capability_Management | Capability_Watcher | Capability_PinCode | Capability_MasterKey | Capability_RatingCheck | Capability_ReadStatus | Capability_StereoVision;
#if defined(NN_DETAIL_PCTL_CONFIG_FULL_CAPABILITY_FOR_RECOVERY)
static const int ServiceCapabilityForRecovery = Capability_All;
NN_STATIC_ASSERT((ServiceCapabilityForRecovery & (Capability_System | Capability_Recovery)) == (Capability_System | Capability_Recovery));
#else
static const int ServiceCapabilityForRecovery = Capability_System | Capability_Recovery;
#endif
static const int ServiceCapabilityFull = Capability_Full;

// 以下の権限は必ず Capability_System か Capability_Application を持つようにする
NN_STATIC_ASSERT((ServiceCapabilityForGeneral & (Capability_Application | Capability_System)) != 0);
NN_STATIC_ASSERT((ServiceCapabilityForSystem & (Capability_Application | Capability_System)) != 0);
NN_STATIC_ASSERT((ServiceCapabilityForAuthentication & (Capability_Application | Capability_System)) != 0);
NN_STATIC_ASSERT((ServiceCapabilityForRecovery & (Capability_Application | Capability_System)) != 0);
NN_STATIC_ASSERT((ServiceCapabilityFull & (Capability_Application | Capability_System)) != 0);

// @brief Capability のチェックを行い、失敗した場合 Result を投げます。
//        variable に値を持つ変数、capability に Capability 値を指定します。
// @details capability に指定する値が複数のビットの組み合わせである場合、
//          その中の1つでも有効であればチェック通過とします。
#define NN_DETAIL_PCTL_CHECK_CAPABILITY(variable, capability) \
    NN_RESULT_THROW_UNLESS(((variable) & (capability)) != 0, nn::pctl::ResultNotPermitted())

}}}}
