﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_TimeSpan.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

namespace
{
    const char ScsiDefault[] = "https://storage.%.scsi.srv.nintendo.net";
    const char PolicyDefault[] = "https://scsi-policy-%.cdn.nintendo.net";
    const char AuthorizationLead[] = "Authorization: Bearer ";
}
// NsaIdToken 認証に必要なヘッダサイズ「
const size_t RequiredBufferSizeForAuthorizationHeader = sizeof(AuthorizationLead) + 3072;
// IO のためのバッファ最小サイズ
static const size_t IoBufferSizeMin = 1024u;
// クライアント証明書
static const size_t DeviceCertBufferSizeMax = 4096u;
// UL/DL の下限速度
// TODO : fwdbg の setting に保存するか検討
static const size_t LowSpeedLimitForTransfer = 8 * 1024; // 8kbps まで対応
// 転送タイムアウトの下限
// TODO : fwdbg の setting に保存するか検討
static const TimeSpan MinimumTransferTimeout = TimeSpan::FromSeconds(10 * 60);

inline TimeSpan GetTransferTimeout(size_t dataSize) NN_NOEXCEPT
{
    auto timeout = TimeSpan::FromSeconds(static_cast<int64_t>(dataSize / (LowSpeedLimitForTransfer / 8)));
    return timeout < MinimumTransferTimeout ? MinimumTransferTimeout : timeout;
}

inline int CreateUrlForRequestSaveDataArchiveInfoList(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives", ScsiDefault);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForRequestSaveDataArchiveInfoListWithApplicationId(char* buffer, size_t bufferSize, ApplicationId appId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/?application_id=%016llx", ScsiDefault, appId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForRequestSaveDataArchiveInfoListWithId(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForRequestStartDownloadSaveDataArchive(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/start_download", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForRequestFinishDownloadSaveDataArchive(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/finish_download", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForStartSaveDataArchiveUpload(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/start_upload", ScsiDefault);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForDeleteSaveDataArchive(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/delete?force=true", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForCreateComponentFile(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/component_files/create", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForCompleteComponentFile(char* buffer, size_t bufferSize, ComponentFileId cfId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/component_files/%llu/finish_upload", ScsiDefault, cfId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForFinishSaveDataArchiveUpload(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/finish_upload", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForGetKeySeedPackage(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/generate_key_seed_package", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForStartSaveDataArchiveDiffUpload(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/start_upload", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForExtendSaveDataArchiveUploadTimeout(char* buffer, size_t bufferSize, SaveDataArchiveId sdaId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/save_data_archives/%llu/extend_upload_timeout", ScsiDefault, sdaId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForUpdateComponentFile(char* buffer, size_t bufferSize, ComponentFileId cfId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/component_files/%llu/update", ScsiDefault, cfId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForGetComponentFileInfo(char* buffer, size_t bufferSize, ComponentFileId cfId) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/component_files/%llu/signed_uri", ScsiDefault, cfId);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateUrlForPolicyInfo(char* buffer, size_t bufferSize, ApplicationId appId, uint32_t version) NN_NOEXCEPT
{
    auto l = nn::util::SNPrintf(buffer, bufferSize, "%s/api/nx/v1/application_policy/%016llx/%05u", PolicyDefault, appId, version);
    NN_SDK_ASSERT(l < static_cast<int>(bufferSize)); // 終端チェック
    return static_cast<int>(l < static_cast<int>(bufferSize) ? l : bufferSize - 1);
}

inline int CreateAuthorizationHeader(char* buffer, size_t bufferSize, const NsaIdToken& nsaIdToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(buffer != nullptr);

    // Authorization ヘッダの準備
    int offset = static_cast<int>(sizeof(AuthorizationLead) - 1);
    NN_SDK_ASSERT(static_cast<int>(bufferSize) > offset);
    std::memcpy(buffer , AuthorizationLead, offset);

    offset += nn::util::Strlcpy(buffer + offset, nsaIdToken.data, static_cast<int>(bufferSize - offset));

    buffer[offset] = '\0';
    NN_SDK_ASSERT(nn::util::Strnlen(buffer, static_cast<int>(bufferSize)) == offset);

    return offset;
}

// fs の exporter を空回ししてサイズとハッシュを取得する API
Result GetExportSizeByDryRun(size_t* outValue, SaveDataChunkDigest* pOutDigest, fs::ISaveDataDivisionExporter* pExporter, fs::SaveDataChunkId id, void* buffer, size_t BufferSize) NN_NOEXCEPT;


}}}} // namespace nn::olsc::srv::transfer
