﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>

#include <cctype>

namespace nn { namespace olsc { namespace srv {

inline void DumpSaveDataArchiveInfo(const SaveDataArchiveInfo& sda) NN_NOEXCEPT
{
    NN_DETAIL_OLSC_TRACE("Dump Sda below.\n");
    NN_DETAIL_OLSC_TRACE("|      field name     |          value          |  necessary  |\n");
    NN_DETAIL_OLSC_TRACE("|          id         |   0x%016llx    | O |\n", sda.id);
    NN_DETAIL_OLSC_TRACE("|         nsaId       |   0x%016llx    | O |\n", sda.nsaId.id);
    NN_DETAIL_OLSC_TRACE("|         appId       |   0x%016llx    | O |\n", sda.applicationId.value);
    NN_DETAIL_OLSC_TRACE("|         uid 0       |   0x%016llx    | O |\n", sda.userId._data[0]);
    NN_DETAIL_OLSC_TRACE("|         uid 1       |   0x%016llx    | O |\n", sda.userId._data[1]);
    NN_DETAIL_OLSC_TRACE("|       deviceId      |   0x%016llx    | O |\n", sda.deviceId);
    NN_DETAIL_OLSC_TRACE("|       dataSize      |   0x%016llx    | O |\n", sda.dataSize);
    NN_DETAIL_OLSC_TRACE("|       commitId      |   0x%016llx    | O |\n", sda.seriesInfo.commitId);
    NN_DETAIL_OLSC_TRACE("|       seriesId      |   0x%016llx    | O |\n", sda.seriesInfo.seriesId);
    switch(sda.status)
    {
        case SaveDataArchiveStatus::Fixed :
            NN_DETAIL_OLSC_TRACE("|        status       |         fixed           | O |\n");
            break;
        case SaveDataArchiveStatus::Uploading :
            NN_DETAIL_OLSC_TRACE("|        status       |       uploading         | O |\n");
            break;
        default :
            NN_DETAIL_OLSC_TRACE("|        status       |        No State         | O |\n");
            break;
    }
    NN_DETAIL_OLSC_TRACE("|       dataSize      |   0x%016llx    | O |\n", sda.dataSize);
    NN_DETAIL_OLSC_TRACE("|launchRequiredVersion|   0x%016llx    | O |\n", sda.launchRequiredVersion);
    NN_DETAIL_OLSC_TRACE("|  numOfPartitions |   0x%016llx    | O |\n", sda.numOfPartitions);
    NN_DETAIL_OLSC_TRACE("|       savedAt       |   0x%016llx    | O |\n", sda.savedAt.value);
    NN_DETAIL_OLSC_TRACE("|     finishedAt      |   0x%016llx    | X |\n", sda.finishedAt.value);
    NN_DETAIL_OLSC_TRACE("|      timeoutAt      |   0x%016llx    | X |\n", sda.timeoutAt.value);
    NN_DETAIL_OLSC_TRACE("|      createdAt      |   0x%016llx    | X |\n", sda.createdAt.value);
    NN_DETAIL_OLSC_TRACE("|      updatedAt      |   0x%016llx    | X |\n", sda.updatedAt.value);
    NN_DETAIL_OLSC_TRACE("|     autobackup      |           0x%d           | X |\n", sda.autoBackup);
    NN_DETAIL_OLSC_TRACE("|    hasThumbnail     |           0x%d           | X |\n", sda.hasThumbnail);
    NN_DETAIL_OLSC_TRACE("\n");
}// NOLINT(impl/function_size)

inline void DumpComponentFileInfo(const ComponentFileInfo& cf) NN_NOEXCEPT
{
    NN_DETAIL_OLSC_TRACE("Dump Cf below.\n");
    NN_DETAIL_OLSC_TRACE("|      field name     |          value          |  necessary  |\n");
    NN_DETAIL_OLSC_TRACE("|          id         |   0x%016llx    | O |\n", cf.id);
    NN_DETAIL_OLSC_TRACE("|         index       |   0x%016llx    | O |\n", cf.clientArgument.chunkId);
    NN_DETAIL_OLSC_TRACE("|  componentFileSize  |   0x%016llx    | O |\n", cf.componentFileSize);
    NN_DETAIL_OLSC_TRACE("|  saveDataChunkSize  |   0x%016llx    | O |\n", cf.saveDataChunkSize);
    for(int i = 0; i < ComponentFileDigest::Size / 8; i++)
    {
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("| componentFileDigest | ");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|                     | ");
        }
        for(int j = 0; j < 8; j++)
        {
            NN_DETAIL_OLSC_TRACE("%02x ", cf.componentFileDigest.data[i * 8 + j]);
        }
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("| O |\n");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|   |\n");
        }
    }

    for(int i = 0; i < SaveDataChunkDigest::Size / 8; i++)
    {
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("| saveDataChunkDigest | ");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|                     | ");
        }
        for(int j = 0; j < 8; j++)
        {
            NN_DETAIL_OLSC_TRACE("%02x ", cf.saveDataChunkDigest.data[i * 8 + j]);
        }
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("| O |\n");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|   |\n");
        }
    }

    switch(cf.type)
    {
        case ComponentFileType::Meta :
        NN_DETAIL_OLSC_TRACE("|         type        |         Meta            | O |\n");
        break;
        case ComponentFileType::Save :
        NN_DETAIL_OLSC_TRACE("|         type        |         Save            | O |\n");
        break;
        default :
        NN_DETAIL_OLSC_TRACE("|         type        |        No Type          | O |\n");
        break;
    }

    switch(cf.status)
    {
        case ComponentFileStatus::Fixed :
            NN_DETAIL_OLSC_TRACE("|        status       |         Fixed           | O |\n");
            break;
        case ComponentFileStatus::Uploading :
            NN_DETAIL_OLSC_TRACE("|        status       |       Uploading         | O |\n");
            break;
        case ComponentFileStatus::HandOver :
            NN_DETAIL_OLSC_TRACE("|        status       |        HandOver         | O |\n");
            break;
        default :
            NN_DETAIL_OLSC_TRACE("|        status       |        No State         | O |\n");
            break;
    }

    auto length = nn::util::Strnlen(cf.url, nn::olsc::srv::ComponentFileInfo::MaxUrlLength);
    for(int i = 0; i < length / 24; i++)
    {
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("|        get_url      | ");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|                     | ");
        }
        for(int j = 0; j < 24; j++)
        {
            NN_DETAIL_OLSC_TRACE("%c", cf.url[i * 24 + j]);
        }
        if(i == 0)
        {
            NN_DETAIL_OLSC_TRACE("| X |\n");
        }
        else
        {
            NN_DETAIL_OLSC_TRACE("|   |\n");
        }
    }

    NN_DETAIL_OLSC_TRACE("|         sdaId       |   0x%016llx    | X |\n", cf.sdaId);
    NN_DETAIL_OLSC_TRACE("|      createdAt      |   0x%016llx    | X |\n", cf.createdAt.value);
    NN_DETAIL_OLSC_TRACE("|      updatedAt      |   0x%016llx    | X |\n", cf.updatedAt.value);
    NN_DETAIL_OLSC_TRACE("\n");
}// NOLINT(impl/function_size)

}}} // namespace nn::olsc::srv
