﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  データの転送ポリシーを扱うインターフェースです。
 */

#include <nn/nn_ApplicationId.h>
#include <nn/olsc/olsc_AsyncRequest.h>
#include <nn/olsc/olsc_DataTypes.h>

namespace nn { namespace olsc {

    /**
    *       @brief      アプリケーションID を指定して対象のアプリケーションのデータ転送ポリシーキャッシュの更新を行います。
    *
    *       @param[out] outAsync    非同期リクエスト。
    *       @param[in]  id          アプリケーションID。
    *
    *       @return     非同期タスクの作成に成功すれば ResultSuccess
    *
    *       @pre
    *           -       インターネット接続要求が受理されている。
    *
    *       @details    指定したプリケーションのデータ転送ポリシーを取得しローカルのキャッシュに永続化します。
    *
    */
    Result RequestUpdateDataTransferPolicy(AsyncRequest* outAsync, ApplicationId id) NN_NOEXCEPT;

    /**
    *       @brief      アプリケーションID を指定して対象のアプリケーションデータの転送ポリシーを取得します。
    *
    *       @param[out] outValue    取得した転送ポリシーの格納先。
    *       @param[in]  id          アプリケーションID。
    *
    *       @return     ポリシーが存在する場合は true。
    *
    *       @pre
    *           -   outValue != nullptr
    *
    *       @details    本体にキャッシュされている転送ポリシーのうち id が一致しているものを取得します。
    *
    *                   この API はシステムにキャッシュされているデータの転送ポリシーを返します。
    *                   サーバの最新の状況を返さないことに注意してください。
    *
    *                   最新の状況が必要な場合は、事前に RequestUpdateDataTransferPolicy を呼び出して
    *                   ローカルのキャッシュを更新するようにしてください。
    *
    */
    bool GetDataTransferPolicy(DataTransferPolicy* outValue, ApplicationId id) NN_NOEXCEPT;

}} // namespace nn::olsc
