﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/ntc/ntc_EnsureNetworkClockAvailabilityMode.h>
#include <nn/ntc/detail/service/ntc_IEnsureNetworkClockAvailabilityService.sfdl.h>
#include <nn/ntc/detail/service/ntc_IStaticService.sfdl.h>
#include <nn/ntc/detail/service/ntc_StaticService.h>

#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace ntc { namespace detail { namespace shim {

/**
 * @brief   ネットワーク時計の有効性確立を非同期に行うためのクラスです
 * @details
 *  EnsureNetworkClockAvailabilityClientImpl クラスのオブジェクトを大域的に配置することはできません。
 */
class EnsureNetworkClockAvailabilityClientImpl
{
NN_DISALLOW_COPY(EnsureNetworkClockAvailabilityClientImpl);
NN_DISALLOW_MOVE(EnsureNetworkClockAvailabilityClientImpl);

public:
    /**
     * @brief   コンストラクタです。
     * @param[in]   eventClearMode                      GetFinishNotificationEvent() で得られるシステムイベントのクリアモードを指定します。
     * @param[in]   ensureNetworkClockAvailabilityMode  有効性確立のモードを指定します。
     * @details
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  EnsureNetworkClockAvailabilityClientImpl クラスのオブジェクトを大域的に配置することはできません。
     */
    EnsureNetworkClockAvailabilityClientImpl(
        nn::os::EventClearMode eventClearMode,
        nn::ntc::EnsureNetworkClockAvailabilityMode ensureNetworkClockAvailabilityMode) NN_NOEXCEPT;

    ~EnsureNetworkClockAvailabilityClientImpl() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク時計の有効性確立の非同期処理を開始します。
     * @return  TBD
     * @details
     *  StartTask() を実行後に GetFinishNotificationEvent() で得られる
     *  システムイベントのシグナルを検知することで、非同期処理の完了を知ることができます。
     *
     *  既に処理中である場合に再度 StartTask() を実行しても、処理は一度しか行われません。
     *
     *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル後であれば、
     *  再度 StartTask() で処理を再実行することができます。
     */
    nn::Result StartTask() NN_NOEXCEPT;

    /**
     * @brief   非同期処理の結果を取得します
     * @return  TBD
     * @details
     *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル後に、
     *  処理の正しい結果が得られるようになります。
     *
     *  StartTask() 実行前や、システムイベントのシグナル前に呼び出しても副作用はありません。
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /**
     * @brief   非同期処理の完了時にシグナルされるシステムイベントを取得します。
     * @details
     *  StartTask() 実行前に呼び出しても副作用はありません。@n
     *  その時のシステムイベントは非シグナル状態です。
     */
    nn::os::SystemEvent& GetFinishNotificationEvent() NN_NOEXCEPT;

    /**
     * @brief   実行中の非同期処理をキャンセルします。
     * @details
     *  StartTask() 実行前に呼び出しても副作用はありません。
     *
     *  キャンセルが正常に受け付けられた場合、
     *  GetFinishNotificationEvent() で得られるシステムイベントがシグナル状態に遷移し、
     *  GetResult() はキャンセルされたことを示す値を返すようになります。
     */
    void Cancel() NN_NOEXCEPT;

    /**
     * @breif   非同期処理が実行中かどうかを返します。
     * @return  非同期処理が実行中かどうか
     * @retval  true    非同期処理が実行中である
     * @retval  false   非同期処理が実行中でない
     */
    bool IsProcessing() NN_NOEXCEPT;

    /**
     * @brief   サーバー時刻を取得します。(EnsureNetworkClockAvailabilityMode_GetServerTime モード専用)
     * @retval  nn::ResultSuccess()                 成功
     * @retval  nn::time::ResultNotImplemented()    対応していない nn::ntc::EnsureNetworkClockAvailabilityMode である場合に返る
     * @retval  nn::time::ResultNotFound()          タスク処理が一度も成功していない
     * @retval  その他                              想定外の失敗
     */
    nn::Result GetServerTime(nn::time::PosixTime* pOutServerTime) NN_NOEXCEPT;

private:
    nn::sf::SharedPointer<nn::ntc::detail::service::IEnsureNetworkClockAvailabilityService> m_pService;
    nn::os::SystemEvent m_FinishEvent;

};

}}}} // nn::ntc::detail::shim
