﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nsd/detail/nsd_Log.h>

namespace nn { namespace nsd { namespace detail { namespace util {

    /**
     * @brief   パスの階層構造を扱うユーティリティクラス
     * @details
     *  Push("setting") を行うと、内部のパスが "/setting" 、
     *  続けて Push("detail") を行うと内部のパスが "/setting/detail" になります。
     *  続けて Pop() を行うと、"/setting" に戻ります。
     *
     * StartArray() はパスに '[' を埋め込みます。
     * EndArray() で末尾の '[' が除去されます。
     *
     *  Compare("/setting/detail") で内部パスの完全一致チェックを行うことができます。
     *
     *  "/setting/detail/ *" などの前方一致は、 CompareFront("/setting/detail", 3) で行ことができます。
     *  第2引数の 3 はスラッシュ数(パスの深さ)です。
     */
    class PathHandler
    {
    public:
        /**
         * @brief   コンストラクタ
         */
        PathHandler() NN_NOEXCEPT:
            m_Depth(0),
            m_ArrayDepth(0)
        {
            std::memset(m_Buffer, 0, sizeof(m_Buffer));
        }

        /**
         * @brief   パスの完全一致チェック
         * @param[in]   path    一致チェックしたいパス文字列
         * @retval  true    一致している
         * @retval  false   一致していない
         */
        bool Compare(const char* path) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(path);
            return nn::util::Strncmp(m_Buffer, path, static_cast<int>(sizeof(m_Buffer))) == 0;
        }

        /**
         * @brief   パスの前方一致チェック
         * @param[in]   path    一致チェックしたいパス文字列
         * @param[in]   depth   一致チェックしたいパスの深さ
         * @retval  true    一致している
         * @retval  false   一致していない
         */
        bool CompareFront(const char* path, int depth) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(path);

            if(depth != m_Depth)
            {
                return false;
            }

            // 終端含まない文字数までの一致確認
            int len = nn::util::Strnlen(path, sizeof(m_Buffer));
            return nn::util::Strncmp(m_Buffer, path, len) == 0;
        }

        /*
         * @brief   パスの末尾一致チェック
         * @param[in]   path    一致チェックしたいパス文字列
         * @retval  true    一致している
         * @retval  false   一致していない
         */
        bool CompareTail(const char* path) const NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(path);

            auto c = std::strrchr(m_Buffer, '/');
            if(c == nullptr)
            {
                return false;
            }

            auto size = sizeof(m_Buffer) - (c - m_Buffer);

            return nn::util::Strncmp(c, path, static_cast<int>(size)) == 0;
        }

        /**
         * @brief   末尾へパスの入れ込み
         * @param[in]   path    パス
         * @param[in]   length  パス文字列の長さ
         * @retval  true    成功
         * @retval  false   失敗（内部パスのサイズが足りなかった）
         */
        bool Push(const char* path, unsigned int length) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_NOT_NULL(path);

            int currentLen = nn::util::Strnlen(m_Buffer, sizeof(m_Buffer));
            int bufferLen = sizeof(m_Buffer) - currentLen - length;
            if(bufferLen <= 0)
            {
                NN_DETAIL_NSD_ERROR("[NSD][PathHandler] path length overflowed.\n");
                return false;
            }

            m_Buffer[currentLen] = '/';
            currentLen++;
            bufferLen--;
            nn::util::Strlcpy(m_Buffer + currentLen, path, bufferLen);

            m_Depth++;
            return true;
        }

        /**
         * @brief   末尾のパスの取り出し
         */
        void Pop() NN_NOEXCEPT
        {
            int currentLen = nn::util::Strnlen(m_Buffer, sizeof(m_Buffer));
            if(m_Buffer[currentLen - 1] == '[')
            {
                // Array 内の要素であれば Pop しない
                return;
            }

            auto c = std::strrchr(m_Buffer, '/');
            if(c == nullptr)
            {
                m_Buffer[0] = '\0';
            }
            else
            {
                *c = '\0';
            }

            m_Depth--;
        }

        int GetPathDepth() const NN_NOEXCEPT
        {
            return m_Depth;
        }

        int GetArrayDepth() const NN_NOEXCEPT
        {
            return m_ArrayDepth;
        }

        bool StartArray() NN_NOEXCEPT
        {
            m_ArrayDepth++;

            int currentLen = nn::util::Strnlen(m_Buffer, sizeof(m_Buffer));
            int bufferLen = sizeof(m_Buffer) - currentLen - 1;
            if(bufferLen <= 0)
            {
                NN_DETAIL_NSD_ERROR("[NSD][PathHandler] path length overflowed.\n");
                return false;
            }

            m_Buffer[currentLen] = '[';
            currentLen++;
            m_Buffer[currentLen] = '\0';

            // NN_SDK_LOG("[StartArray] %s\n", m_Buffer);

            return true;
        }

        void EndArray() NN_NOEXCEPT
        {
            m_ArrayDepth--;

            int currentLen = nn::util::Strnlen(m_Buffer, sizeof(m_Buffer));
            currentLen--;

            NN_SDK_ASSERT(m_Buffer[currentLen] == '[', "%s, currentLen:%d", m_Buffer, currentLen );
            m_Buffer[currentLen] = '\0';

            // NN_SDK_LOG("[EndArray] %s\n", m_Buffer);
        }

    private:
        char m_Buffer[256]; // TORIAZEU:固定
        int m_Depth;

        int m_ArrayDepth;
        //int m_ArrayElementIndex;
    };

    class PathEntry
    {
    public:
        explicit PathEntry(const char* path) NN_NOEXCEPT:
            m_Path(path),
            m_isMatched(false)
        {}

        bool Match(const PathHandler& pathHandler) NN_NOEXCEPT
        {
            if(pathHandler.Compare(m_Path))
            {
                m_isMatched = true;
                return true;
            }
            else
            {
                return false;
            }
        }

        nn::Result CheckAccepted() const NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(
                m_isMatched, nn::nsd::ResultRequiredItemNotFound());
            NN_RESULT_SUCCESS;
        }

    private:
        const char* m_Path;
        bool m_isMatched;
    };

}}}} // nn::nsd::detail::util
