﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief IPC サーバー内部利用向けのクラス宣言
 */

#pragma once

#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/nsd/nsd_BackboneSettingsTypes.h>

#include <nn/os/os_Mutex.h>

namespace nn { namespace nsd { namespace detail {

    /**
     * @brief FQDN のリゾルバー
     * @details
     *  SetSaveData() で注入したセーブデータに従って FQDN を解決するリゾルバーです。
     *
     *  利用前に必ず SetSaveData() でサービスディスカバリのセーブデータを設定してください。
     *
     *  すべてのメンバ API は内部で排他処理を行っています。
     */
    class FqdnResolver
    {
         NN_DISALLOW_COPY(FqdnResolver);
         NN_DISALLOW_MOVE(FqdnResolver);

    public:

        /**
         * @brief   コンストラクタです。
         */
        FqdnResolver() NN_NOEXCEPT;

        /**
         * @brief   サービスディスカバリのセーブデータを設定します。
         * @param[in]   pSaveData   セーブデータ
         *
         * @pre
         *  - pSaveData != nullptr
         *
         * @details
         *  他のメンバ関数利用前に必ずこの関数でセーブデータを設定してください。
         *
         *  設定したセーブデータのバッファは本クラスの利用終了まで呼び出し元で保持しておく必要があります。
         */
        Result SetSaveData(SaveData* pSaveData) NN_NOEXCEPT;

        /**
         * @brief FQDN の解決
         * @param[out]  pOut    解決された FQDN
         * @param[in]   fqdn    解決したい FQDN
         *
         * @retresult
         *      @handleresult{ResultInvalidSettingVersion}
         *      @handleresult{ResultInvalidSettingData}
         *      @handleresult{ResultFqdnTooLong, 処理結果の FQDN が長すぎます}
         *      @handleresult{ResultFqdnIsNotNullTerminated, 入力の FQDN が NULL 終端していません}
         *      @handleresult{ResultFsError, ファイルシステムエラー}
         * @endretresult
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOut != nullptr
         * @post
         *  - pOut は NULL 終端している
         *
         * @details
         *  - 解決する必要がない場合は入力と同じ FQDN が pOut へ出力されます。
         */
        Result Resolve(
            Fqdn *pOut,
            const Fqdn& fqdn) NN_NOEXCEPT;

        /**
         * @brief FQDN の解決
         * @param[out]  pOutInnerResult     内部処理の結果
         * @param[out]  pOut                解決された FQDN
         * @param[in]   fqdn                解決したい FQDN
         *
         * @retresult
         *      @handleresult{ResultFqdnTooLong, 処理結果の FQDN が長すぎます}
         *      @handleresult{ResultFqdnIsNotNullTerminated, 入力の FQDN が NULL 終端していません}
         * @endretresult
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOutInnerResult != nullptr
         *  - pOut != nullptr
         *
         * @details
         *  インポート済設定の互換性がないことに起因するエラーを隠蔽し、
         *  入力の FQDN が不正な場合にしかエラーを返さない版.
         *  FQDN 解決失敗時には名前解決できない FQDN を pOut へ格納.
         */
        Result ResolveEx(
            Result* pOutInnerResult,
            Fqdn *pOut,
            const Fqdn& fqdn) NN_NOEXCEPT;

        /**
         * @brief   NAS の認証・トークン取得リクエスト用の FQDN を取得します。
         * @param[out]  pOut    FQDN
         *
         * @retresult
         *      @handleresult{ResultInvalidSettingVersion}
         *      @handleresult{ResultInvalidSettingData}
         *      @handleresult{ResultNotFound, 設定が存在しません}
         *      @handleresult{ResultFsError, ファイルシステムエラー}
         * @endretresult
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOut != nullptr
         */
        Result GetNasRequestFqdn(Fqdn* pOut) NN_NOEXCEPT;

        /**
         * @brief   NAS の認証・トークン取得リクエスト用の FQDN を取得します。
         * @param[out]  pOutInnerResult     内部処理の結果
         * @param[out]  pOut                FQDN
         *
         * @return ResultSuccess
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOutInnerResult != nullptr
         *  - pOut != nullptr
         *
         * @details
         *  インポート済設定の互換性がない場合には成功を返し、
         *  名前解決できない FQDN を pOut へ格納します。
         *
         *  FqdnResolver が不正状態の場合も成功を返し、
         *  名前解決できない FQDN を pOut へ格納します。
         */
        Result GetNasRequestFqdnEx(
            Result* pOutInnerResult,
            Fqdn* pOut) NN_NOEXCEPT;

        /**
         * @brief   NAS の API 用の FQDN を取得します。
         * @param[out]  pOut    FQDN
         *
         * @retresult
         *      @handleresult{ResultInvalidSettingVersion}
         *      @handleresult{ResultInvalidSettingData}
         *      @handleresult{ResultNotFound, 設定が存在しません}
         *      @handleresult{ResultFsError, ファイルシステムエラー}
         * @endretresult
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOut != nullptr
         */
        Result GetNasApiFqdn(Fqdn* pOut) NN_NOEXCEPT;

        /**
         * @brief   NAS の API 用の FQDN を取得します。
         * @param[out]  pOutInnerResult     内部処理の結果
         * @param[out]  pOut                FQDN
         *
         * @return  ResultSuccess
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOutInnerResult != nullptr
         *  - pOut != nullptr
         *
         * @details
         *  インポート済設定の互換性がない場合には成功を返し、
         *  名前解決できない FQDN を pOut へ格納します。
         *
         *  FqdnResolver が不正状態の場合も成功を返し、
         *  名前解決できない FQDN を pOut へ格納します。
         */
        Result GetNasApiFqdnEx(
            Result* pOutInnerResult,
            Fqdn* pOut) NN_NOEXCEPT;

        /**
         * @brief   NAS サービスを利用するための設定を取得します。
         * @param[out]  pOutNasServiceSetting   設定格納先
         * @param[in]   nasServiceName          設定を取得したいサービスの名前
         *
         * @retresult
         *      @handleresult{ResultInvalidSettingVersion}
         *      @handleresult{ResultInvalidSettingData}
         *      @handleresult{ResultNotFound, 指定の NasServiceName に該当する設定が存在しません}
         *      @handleresult{ResultFsError, ファイルシステムエラー}
         * @endretresult
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOutNasServiceSetting != nullptr
         */
        Result GetNasServiceSetting(
            NasServiceSetting* pOutNasServiceSetting,
            const NasServiceName& nasServiceName) NN_NOEXCEPT;

        /**
         * @brief   NAS サービスを利用するための設定を取得します。
         * @param[out]  pOutInnerResult         内部処理の結果
         * @param[out]  pOutNasServiceSetting   設定格納先
         * @param[in]   nasServiceName          設定を取得したいサービスの名前
         *
         * @return  ResultSuccess
         *
         * @pre
         *  - SetSaveData でセーブデータが設定されている
         *  - pOutInnerResult != nullptr
         *  - pOutNasServiceSetting != nullptr
         *
         * @details
         *  インポート済設定の互換性がない場合には成功を返し、
         *  不正な設定値を出力します。
         *
         *  FqdnResolver が不正状態の場合も成功を返し、
         *  不正な設定値を出力します。
         */
        Result GetNasServiceSettingEx(
            Result* pOutInnerResult,
            NasServiceSetting* pOutNasServiceSetting,
            const NasServiceName& nasServiceName) NN_NOEXCEPT;

        nn::Result GetSettingName(SettingName* pOut) NN_NOEXCEPT;
        nn::Result GetEnvironmentIdentifier(EnvironmentIdentifier* pOut) NN_NOEXCEPT;

        /**
         * @brief   FQDN リゾルバを無効化します。
         * @details これが呼ばれると以降の FqdnResolver の 名前解決関連の関数が
         * セットしたエラーを返すようになります。
         */
        void SetFatalError( nn::Result result ) NN_NOEXCEPT;

        /**
         * @brief   起動時のサービスディスカバリ設定の存在フラグセット
         * @details 起動時にサービスディスカバリ設定が存在したかをセットします。
         *          このフラグの決定は起動時にのみ行います。
         */
        void SetSettingPresent( bool isExisted ) NN_NOEXCEPT
        {
            m_IsSettingPresent = isExisted;
        }

        /**
         * @brief   現在のセーブデータの取得
         * @details 現在のセーブデータへのポインタを取得する。
         */
        Result GetSaveData( SaveData** pSaveData ) NN_NOEXCEPT;

    private:
        nn::os::Mutex m_LockOfApi;

    protected:
        SaveData* m_pSaveData;
        nn::Result m_FatalError;  // 通常 ResultSuccess() で、それ以外だと以降の Resolve()等が失敗する
        bool m_IsSettingPresent; // 起動時にセーブデータが存在なら true、それ以外(読み込みエラーも含む)は false
    };

    FqdnResolver* GetFqdnResolverPtr() NN_NOEXCEPT;

}}} // nn::nsd::detail
