﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/util/nsd_PathHandler.h>
#include <nn/nsd/detail/nsd_Log.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_NAMESPACE             nne::rapidjson
#define RAPIDJSON_NAMESPACE_BEGIN       namespace nne { namespace rapidjson {
#define RAPIDJSON_NAMESPACE_END         }}
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(preprocessor/const)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(preprocessor/const)
#include <rapidjson/reader.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif


namespace nn { namespace nsd { namespace detail { namespace json {

/** @brief メモリのアロケートを禁止したアロケータ
    @details
        RapidJSON に指定するアロケータの実装。
        RapidJSON を In-situ パースのモードで使用するとメモリ確保が不要になるため、 Free 以外の内部実装が呼ばれることはありません。
 */
class NoAllocatorForRapidJson
{
public:
    static const bool kNeedFree = false;

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_ABORT(
            "[nn::nsd] -----------------------------------------------\n"
            "  ABORT: Memory allocation not allowed: %zu bytes\n", size);
    }
    void* Realloc(void* originalPtr, size_t originalSize, size_t newSize) NN_NOEXCEPT
    {
        NN_UNUSED(originalPtr);
        NN_UNUSED(originalSize);
        NN_UNUSED(newSize);
        NN_ABORT(
            "[nn::nsd] -----------------------------------------------\n"
            "  ABORT: Memory re-allocation not allowed: %zu -> %zu bytes\n", originalSize, newSize);
    }
    static void Free(void *ptr) NN_NOEXCEPT
    {
        // NOTE 使い捨てのアロケータのため free しない
        NN_UNUSED(ptr);
    }
};

/**
 * @brief   RapidJSON の InputStrem
 * @details
 *  このクラスは抽象クラスです。
 *  InputStreamForRapidJson::RequestBufferImpl を継承クラスで実装してください。
 */
class InputStreamForRapidJson
{
public:
    typedef char Ch;

private:

    // document buffer
    Ch *m_DocumentBuffer;
    const size_t m_DocumentBufferSize;
    size_t m_PositionInDocumentBuffer;
    size_t m_TotalDocumentBufferedBytes;
    size_t m_DocumentBufferedBytes;

    // string buffer
    Ch *m_StringBuffer;
    const size_t m_StringBufferSize;
    size_t m_PositionInStringBuffer;

    nn::Result m_Result;

public:
    /**
     * @brief   コンストラクタです
     * @param[in]   stringBuffer            文字列解析結果を入れるバッファ
     * @param[in]   stringBufferSize        stringBuffer のサイズ(これより長い文字列は切り詰められます)
     * @param[in]   documentBuffer          一度の解析処理で扱う JSON ドキュメントのバッファ
     * @param[in]   documentBufferSize      documentBuffer のサイズ
     */
    InputStreamForRapidJson(
        Ch* stringBuffer, size_t stringBufferSize,
        Ch* documentBuffer, size_t documentBufferSize) NN_NOEXCEPT :
        m_DocumentBuffer(documentBuffer),
        m_DocumentBufferSize(documentBufferSize),
        m_PositionInDocumentBuffer(0),
        m_TotalDocumentBufferedBytes(0),
        m_DocumentBufferedBytes(0),
        m_StringBuffer(stringBuffer),
        m_StringBufferSize(stringBufferSize),
        m_PositionInStringBuffer(0),
        m_Result(nn::ResultSuccess())
    {
    }

    nn::Result GetResult() const
    {
        return m_Result;
    }

protected:
    /**
     * @brief   解析対象のバッファを与える関数です。バッファがなくなるたびによばれます。
     *
     * @param[out]  pOutActualRead  読み込んだ実際のファイルサイズ
     * @param[out]  buffer          読み込みバッファ
     * @param[in]   bufferSize      buffer のサイズ
     *
     * @details
     *  buffer へ最大 bufferSize 分、解析対象の JSON ドキュメントをコピーし、
     *  pOutActualRead へコピーしたサイズを格納してください。
     *
     *  pOutActualRead に 0 を代入した場合は解析処理を終えます。
     *  buffer が NULL 終端した場合、その JSON ドキュメントで解析を終えます。
     *
     *  この関数で返す nn::Result は nn::nsd::detail::json::Parse() の返り値としてハンドリングできます。
     */
    virtual nn::Result RequestBufferImpl(size_t *pOutActualRead, void *buffer, size_t bufferSize) NN_NOEXCEPT = 0;

public:
    // 現在位置の文字を返し、位置は変えない。
    Ch Peek() NN_NOEXCEPT
    {
        if (m_PositionInDocumentBuffer >= m_DocumentBufferedBytes)
        {
            NN_SDK_ASSERT(m_PositionInDocumentBuffer == m_DocumentBufferedBytes);
            if (!RequestBuffer())
            {
                return '\0';
            }
        }
        return m_DocumentBuffer[m_PositionInDocumentBuffer];
    }
    // 現在位置の文字を返し、位置を進める。
    Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        m_PositionInDocumentBuffer++;
        return c;
    }
    // 現在位置を返す。
    size_t Tell() NN_NOEXCEPT
    {
        return (m_TotalDocumentBufferedBytes - m_DocumentBufferedBytes) + m_PositionInDocumentBuffer;
    }

    void Flush() NN_NOEXCEPT
    {
    }

    Ch* PutBegin() NN_NOEXCEPT
    {
        m_PositionInStringBuffer = 0;
        std::memset(static_cast<void*>(m_StringBuffer), 0, m_StringBufferSize);
        return m_StringBuffer;
    }
    void Put(Ch c) NN_NOEXCEPT
    {
        if(m_PositionInStringBuffer < m_StringBufferSize)
        {
            m_StringBuffer[m_PositionInStringBuffer] = c;
        }
        // サイズ溢れ検知のためバッファに入らなくてもlengthは増やす
        m_PositionInStringBuffer++;
    }
    size_t PutEnd(Ch* begin) NN_NOEXCEPT
    {
        NN_UNUSED( begin );
        return static_cast<size_t>( m_PositionInStringBuffer );
    }

private:
    // JSON ドキュメントのバッファ要求。true だとバッファをもらえた。
    // Peek() でドキュメントの文字列要求があったときに、手元に JSON ドキュメントのバッファがなければ呼ばれる。
    bool RequestBuffer() NN_NOEXCEPT
    {
        m_PositionInDocumentBuffer = 0;
        m_Result = RequestBufferImpl(&m_DocumentBufferedBytes, m_DocumentBuffer, m_DocumentBufferSize);
        if(m_Result.IsFailure())
        {
            return false;
        }
        m_TotalDocumentBufferedBytes += m_DocumentBufferedBytes;
        return m_DocumentBufferedBytes > 0;
    }
};

/** @brief RapidJSON のエラーを Result に置き換える関数
 */
inline Result HandleParseResult(nne::rapidjson::ParseResult r) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!r, nn::ResultSuccess()); // エラーでないなら即終了
    NN_DETAIL_NSD_INFO_V1("JsonParser: RapidJson returns %d, at offset=%zu\n", r.Code(), r.Offset());
    switch (r.Code())
    {
    // ドキュメントがない
    // 根要素が複数
    case nne::rapidjson::kParseErrorDocumentEmpty:
    case nne::rapidjson::kParseErrorDocumentRootNotSingular:
        return ResultJsonDocumentError();
    // オブジェクトなのに名前がない
    // オブジェクトなのに : がない
    // オブジェクトなのに , や } がない
    // 配列なのに , や ] がない
    // 文字列なのに " で囲まれていない
    // 特定できない文法上のエラー
    case nne::rapidjson::kParseErrorObjectMissName:
    case nne::rapidjson::kParseErrorObjectMissColon:
    case nne::rapidjson::kParseErrorObjectMissCommaOrCurlyBracket:
    case nne::rapidjson::kParseErrorArrayMissCommaOrSquareBracket:
    case nne::rapidjson::kParseErrorStringMissQuotationMark:
    case nne::rapidjson::kParseErrorUnspecificSyntaxError:
        return ResultJsonSyntaxError();
    // 不正な数値
    // \uXXXX が不正
    // サロゲートが不正
    // エスケープ文字が不正
    // 文字列が不正なエンコード
    // 数値が大きすぎる
    // 小数のフォーマットが変, 仮数部がない
    // 小数のフォーマットが変, 指数部がない
    case nne::rapidjson::kParseErrorValueInvalid:
    case nne::rapidjson::kParseErrorStringUnicodeEscapeInvalidHex:
    case nne::rapidjson::kParseErrorStringUnicodeSurrogateInvalid:
    case nne::rapidjson::kParseErrorStringEscapeInvalid:
    case nne::rapidjson::kParseErrorStringInvalidEncoding:
    case nne::rapidjson::kParseErrorNumberTooBig:
    case nne::rapidjson::kParseErrorNumberMissFraction:
    case nne::rapidjson::kParseErrorNumberMissExponent:
        return ResultJsonContentError();
    // 処理が中断された
    case nne::rapidjson::kParseErrorTermination:
        return ResultJsonParsingTerminated();
    // 未知のエラー
    default:
        return ResultJsonEnvironmentError();
    }
}

class JsonEventAccepter
{
public:
    JsonEventAccepter():
        m_Result(nn::ResultSuccess())
    {}

    /*
    virtual bool StartObject() NN_NOEXCEPT = 0;
    virtual bool EndObject(nne::rapidjson::SizeType memberCount) NN_NOEXCEPT = 0;
    virtual bool Null() NN_NOEXCEPT = 0;
    virtual bool Bool(bool b) NN_NOEXCEPT = 0;
    virtual bool Int(int i) NN_NOEXCEPT = 0;
    virtual bool Uint(unsigned u) NN_NOEXCEPT = 0;
    virtual bool Int64(int64_t i) NN_NOEXCEPT = 0;
    virtual bool Uint64(uint64_t u) NN_NOEXCEPT = 0;
    virtual bool Double(double d) NN_NOEXCEPT = 0;
    virtual bool String(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT = 0;
    virtual bool Key(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT = 0;
    virtual bool StartArray() NN_NOEXCEPT = 0;
    virtual bool EndArray(nne::rapidjson::SizeType elementCount) NN_NOEXCEPT = 0;
    */

    virtual nn::Result AcceptString(
        const char* value, size_t length, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        NN_UNUSED(length);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptNull(const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptBool(
        bool value, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptInt64(
        int64_t value, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptUint64(
        uint64_t value, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptDouble(
        double value, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptJsonKey(const char* key, size_t length, const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(key);
        NN_UNUSED(length);
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptEndObject(const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result AcceptEndArray(const util::PathHandler& pathHandler) NN_NOEXCEPT
    {
        NN_UNUSED(pathHandler);
        NN_RESULT_SUCCESS;
    }
    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }

private:
    nn::Result m_Result;
};

/**
 * @brief   rapidjson イベントハンドラのベースクラス
 * @details
 *  抽象クラスです。
 *  StartObject() など JSON 要素を解析する純粋仮想関数を継承先で実装してください。
 *  bool を返す関数は true を返すと処理を続行、 false を返すと処理を中止します。
 *
 *  m_Result に nn::Result を代入することで nn::nsd::detail::json::Parse() の返り値でハンドリングできます。
 */
class RapidJsonEventHandler
{
public:
    explicit RapidJsonEventHandler(JsonEventAccepter* pJsonEventHandler) NN_NOEXCEPT:
        m_Result(nn::ResultSuccess()),
        m_pJsonEventHandler(pJsonEventHandler)
    {}

    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }
protected:
    nn::Result m_Result; //!< 継承先で書き換え可能なリザルトです。 nn::nsd::detail::json::Parse() の返り値でハンドリングできます。
    JsonEventAccepter* m_pJsonEventHandler;
    nn::nsd::detail::util::PathHandler m_PathHandler;

public:
    bool StartObject() NN_NOEXCEPT
    {
        return true;
    }
    bool EndObject(nne::rapidjson::SizeType memberCount) NN_NOEXCEPT
    {
        NN_UNUSED(memberCount);
        m_Result = m_pJsonEventHandler->AcceptEndObject(m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Null() NN_NOEXCEPT
    {
        m_Result = m_pJsonEventHandler->AcceptNull(m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Bool(bool b) NN_NOEXCEPT
    {
        m_Result = m_pJsonEventHandler->AcceptBool(b, m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Int(int i) NN_NOEXCEPT
    {
        return this->Int64(i); // TORIAEZU
    }
    bool Uint(unsigned u) NN_NOEXCEPT
    {
        return this->Uint64(u); // TORIAEZU
    }
    bool Int64(int64_t i) NN_NOEXCEPT
    {
        m_Result = m_pJsonEventHandler->AcceptInt64(i, m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Uint64(uint64_t u) NN_NOEXCEPT
    {
        m_Result = m_pJsonEventHandler->AcceptUint64(u, m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Double(double d) NN_NOEXCEPT
    {
        m_Result = m_pJsonEventHandler->AcceptDouble(d, m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool String(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT
    {
        NN_UNUSED(copy);
        m_Result = m_pJsonEventHandler->AcceptString(str, length, m_PathHandler);
        m_PathHandler.Pop();
        return m_Result.IsSuccess();
    }
    bool Key(const char* str, nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT
    {
        NN_UNUSED(copy);
        if(!m_PathHandler.Push(str, length))
        {
            return false;
        }

        m_Result = m_pJsonEventHandler->AcceptJsonKey(str, length, m_PathHandler);
        return m_Result.IsSuccess();
    }
    bool StartArray() NN_NOEXCEPT
    {
        m_PathHandler.StartArray();
        return true;
    }
    bool EndArray(nne::rapidjson::SizeType elementCount) NN_NOEXCEPT
    {
        NN_UNUSED(elementCount);

        m_Result = m_pJsonEventHandler->AcceptEndArray(m_PathHandler);

        m_PathHandler.EndArray();
        m_PathHandler.Pop();

        return m_Result.IsSuccess();
    }
};

/**
 * @brief   JSON 解析を行います。
 * @param[in]   stream      解析対象となる JSON ドキュメントのインプットストリーム. InputStreamForRapidJson を継承したクラスを指定してください.
 * @param[in]   handler     JSON ドキュメントの各要素を解析するためのイベントハンドラ. JsonEventHandler を継承したクラスを指定してください.
 *
 * @details
 *  エラーハンドリングの優先順序は、stream , handler , JSON の構文解析エラー、の順です。
 */
inline nn::Result Parse(
    nn::nsd::detail::json::InputStreamForRapidJson* stream,
    nn::nsd::detail::json::JsonEventAccepter* accepter)
{
    typedef nne::rapidjson::GenericReader<nne::rapidjson::UTF8<>, nne::rapidjson::UTF8<>, json::NoAllocatorForRapidJson> Reader;
    NoAllocatorForRapidJson allocator;

    nn::nsd::detail::json::RapidJsonEventHandler handler(accepter);
    auto result = Reader(&allocator, 1u).Parse<
        nne::rapidjson::kParseInsituFlag | nne::rapidjson::kParseStopWhenDoneFlag | nne::rapidjson::kParseValidateEncodingFlag>
            (*stream, handler);

    NN_RESULT_DO(stream->GetResult());
    NN_RESULT_DO(handler.GetResult());
    NN_RESULT_DO(HandleParseResult(result));
    NN_RESULT_SUCCESS;
}

/**
 * @brief   nn::util::Base64::Status を NSD のリザルトへ変換します
 */
inline nn::Result HandleBase64Result(nn::util::Base64::Status status) NN_NOEXCEPT
{
    switch(status)
    {
    case nn::util::Base64::Status_BadData:
        NN_RESULT_THROW( nn::nsd::ResultBase64DecodeBadData() );
    case nn::util::Base64::Status_BufferFull:
        NN_RESULT_THROW( nn::nsd::ResultBase64DecodeOutOfMemory() );
    case nn::util::Base64::Status_Success:
        NN_RESULT_SUCCESS;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   rapidjson を利用した Base64UrlSafe エンコードされた JSON ドキュメント向けの解析用インプットストリームです。
 * @tparam  StringBufferSize        文字列解析結果を入れるバッファサイズ(これより長い文字列は切り詰められます)
 * @tparam  DocumentBufferSize      一度の解析処理で扱う JSON ドキュメントのバッファサイズ
 * @details
 *  テンプレート引数で渡したサイズのバッファをメンバ変数で定義します。
 *
 *  Base64UrlSafe エンコードされた JSON ドキュメントを解析する rapidjson のインプットストリームです。
 *  rapidjson の EvnetHandler と組み合わせて、nn::nsd::detail::json::Parse() により JSON を解析できます。
 *
 *  JSON 解析そのもののエラーハンドリングは、 nn::nsd::detail::json::Parse() の返り値で行ってください。
 */
template <size_t StringBufferSize, size_t DocumentBufferSize>
class Base64UrlSafeEncodedJsonInputStream : public nn::nsd::detail::json::InputStreamForRapidJson
{
    NN_DISALLOW_COPY(Base64UrlSafeEncodedJsonInputStream);
    NN_DISALLOW_MOVE(Base64UrlSafeEncodedJsonInputStream);

private:
    char m_StringBuffer[StringBufferSize];
    char m_DocumentBuffer[DocumentBufferSize];

    const char* m_pBase64DecodeSrc;
    const size_t m_base64DecodeSrcSize;
    size_t m_PositionInBase64DecodeSrc;

public:
    /**
     * @brief   コンストラクタ
     * @param[in]   pBase64DecodeSrc        Base64Url エンコードされた文字列バッファ
     * @param[in]   base64DecodeSrcSize     pBase64DecodeSrc のサイズ
     * @details
     */
    Base64UrlSafeEncodedJsonInputStream(const char* pBase64DecodeSrc, size_t base64DecodeSrcSize):
        nn::nsd::detail::json::InputStreamForRapidJson(
            m_StringBuffer, StringBufferSize,
            m_DocumentBuffer, DocumentBufferSize),
        m_pBase64DecodeSrc(pBase64DecodeSrc),
        m_base64DecodeSrcSize(base64DecodeSrcSize),
        m_PositionInBase64DecodeSrc(0)
     {}

protected:
    nn::Result RequestBufferImpl(size_t *pOutActualRead, void *buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        std::memset(buffer, 0, bufferSize);

        if(m_PositionInBase64DecodeSrc == m_base64DecodeSrcSize)
        {
            *pOutActualRead = 0; // JSON パースを終えさせる
            NN_RESULT_SUCCESS;
        }

        // m_Base64DecodeBuffer のサイズにあうだけ base64デコードして buffer へ詰める
        size_t decodeLen = static_cast<int>(bufferSize - 2) / 3 * 4;
        if(m_PositionInBase64DecodeSrc + decodeLen > m_base64DecodeSrcSize)
        {
            decodeLen = m_base64DecodeSrcSize - m_PositionInBase64DecodeSrc;
        }

        size_t outDecodeNum;
        auto result = nn::util::Base64::FromBase64String(
            &outDecodeNum, buffer, bufferSize,
            m_pBase64DecodeSrc + m_PositionInBase64DecodeSrc, decodeLen, nn::util::Base64::Mode_UrlSafe);

        if(result == nn::util::Base64::Status_Success)
        {
            m_PositionInBase64DecodeSrc += decodeLen;
            *pOutActualRead = outDecodeNum;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return HandleBase64Result(result);
        }
    }
};

}}}} // nn::nsd::detail::json
