﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/es/es_Types.h>
#include <nn/ns/ns_TicketApi.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns { namespace srv {
    class TicketManager
    {
    public:
        explicit TicketManager(IntegratedContentManager* pIntegrated, ApplicationRecordDatabase* pRecordDb, RequestServer* requestServer) NN_NOEXCEPT : m_pIntegrated(pIntegrated), m_pRecordDb(pRecordDb), m_RequestServer(requestServer), m_Mutex(false) {};

        void SetNextAvailableTimeForDownloadApplicationPrepurchasedRights(TimeSpan timeSpan) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);
            m_NextAvailableTimeForDownloadApplicationPrepurchasedRights = timeSpan;
        }

        TimeSpan GetNextAvailableTimeForDownloadApplicationPrepurchasedRights() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);
            return m_NextAvailableTimeForDownloadApplicationPrepurchasedRights;
        }

        Result CheckAccountRights(es::AccountId accountId) const NN_NOEXCEPT;
        Result CheckTicketRights(es::RightsIdIncludingKeyId rightsId) const NN_NOEXCEPT;
        Result CheckPrepurchaseRights(es::RightsIdIncludingKeyId rightsId) const NN_NOEXCEPT;
        Result VerifyContentRights(const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex) const NN_NOEXCEPT;
        Result VerifyContentRights(const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result HasContentRights(bool* hasRights, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex) const NN_NOEXCEPT;
        Result HasContentRights(bool* hasRights, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT;

        Result RequestDownloadApplicationPrepurchasedRights(sf::Out<sf::NativeHandle> outHandle, sf::Out<nn::sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetApplicationTicketInfo(nn::sf::Out<ns::ApplicationTicketInfo> outValue, ncm::ApplicationId id) NN_NOEXCEPT;

        Result GetRightsId(es::RightsIdIncludingKeyId* outValue, Bit64 contentMetaId, ncm::ContentType contentType) NN_NOEXCEPT;
        bool IsPrepurchased(es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT;
    private:
        Result VerifyFsRightsId(const fs::RightsId& fsRightsId) const NN_NOEXCEPT;

        IntegratedContentManager* m_pIntegrated;
        ApplicationRecordDatabase* m_pRecordDb;
        RequestServer* m_RequestServer;
        TimeSpan m_NextAvailableTimeForDownloadApplicationPrepurchasedRights;
        os::Mutex m_Mutex;
    };
}}}
