﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/fs_RightsId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ns/srv/ns_ApplicationStoragePriority.h>
#include "ns_OsUtil.h"

namespace nn { namespace ns { namespace srv {
    class IntegratedContentManager
    {
    public:
        void Register(ncm::StorageId storageId, ncm::ContentMetaDatabase&& db, ncm::ContentStorage&& storage) NN_NOEXCEPT;
        void Unregister(ncm::StorageId storageId) NN_NOEXCEPT;
        int ListInstalledApplication(ncm::ApplicationContentMetaKey outValue[], int count) const NN_NOEXCEPT;
        int ListInstalledApplicationContent(ncm::ContentMetaKey outValue[], int count, ncm::ApplicationId id, ncm::ContentMetaType metaType = ncm::ContentMetaType::Unknown, ncm::ContentInstallType installType = ncm::ContentInstallType::Full) const NN_NOEXCEPT;
        int ListContent(ncm::ContentMetaKey outValue[], int count, ncm::StorageId storageId = ncm::StorageId::Any, ncm::ContentMetaType metaType = ncm::ContentMetaType::Unknown, ncm::ContentInstallType installType = ncm::ContentInstallType::Full) const NN_NOEXCEPT;
        Result GetContentId(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex) const NN_NOEXCEPT;
        Result GetContentIdByStorageId(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result CalculateOccupiedSize(int64_t* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result DeleteContentMeta(const ncm::ContentMetaKey& key, const ncm::ContentMetaKey* latest = nullptr, ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;
        Result DeleteDuplicatedContentMeta(const ncm::ContentMetaKey& key) NN_NOEXCEPT;
        Result GetContentPath(ncm::Path* outValue, ncm::ContentId contentId) const NN_NOEXCEPT;
        Result GetContentPathByStorageId(ncm::Path* outValue, ncm::ContentId contentId, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result HasContentMetaKey(bool* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result HasContentMetaKey(bool* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result GetContentMetaStorage(ncm::StorageId* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result GetInstalledContentMetaStorage(ncm::StorageId* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result GetLatest(ncm::ContentMetaKey* outValue, nn::Bit64 id) const NN_NOEXCEPT;
        Result Get(size_t* outSize, void* outValue, size_t bufferSize, const ncm::ContentMetaKey key) const NN_NOEXCEPT;
        Result GetSize(size_t* outSize, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result GetRequiredSystemVersion(uint32_t* out, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result GetPatchId(ncm::PatchId* out, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        bool IsRegisteredStorage(ncm::StorageId storageId) const NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT;

        Result GetContentIdFileSize(int64_t* pOutSize, ncm::ContentId contentId, ncm::StorageId targetStorage) const NN_NOEXCEPT;
        Result ListContentInfo(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset) const NN_NOEXCEPT;
        Result ListContentInfo(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset, ncm::StorageId targetStorage) const NN_NOEXCEPT;
        Result GetContentInfo(ncm::ContentInfo* pOutInfo, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t programIndex, ncm::StorageId targetStorage) const NN_NOEXCEPT;
        Result ReadContentIdFile(void* buffer, size_t bufferSize, ncm::ContentId contentId, int64_t offset, ncm::StorageId targetStorage) const NN_NOEXCEPT;
        Result WriteContentIdFile(ncm::ContentId contentId, int64_t offset, const void* buffer,size_t bufferSize, ncm::StorageId targetStorage) NN_NOEXCEPT;
        Result GetContentIdFileHash(ncm::Hash* pOutHash, ncm::ContentId contentId, ncm::ContentId metaContentId, ncm::StorageId targetStorage) const NN_NOEXCEPT;

        Result GetContentRightsId(fs::RightsId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result GetContentRightsId(fs::RightsId* outRightsId, uint8_t* outKeyGeneration, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT;
        Result CleanupOrphanContents() NN_NOEXCEPT;
        Result CleanupAllPlaceHolder(ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;
        Result CleanupFragments(ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;
        Result RepairInvalidFileAttribute(ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;

        int GetRegisteredStorages(ncm::StorageId* outValue, int numStorage) const NN_NOEXCEPT;

        Result GetRequiredApplicationVersion(uint32_t* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result GetRequiredApplicationVersionByStorageId(uint32_t* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) const NN_NOEXCEPT;

        Result HasOrphanContents(bool* outValue) const NN_NOEXCEPT;

        Result EstimateRequiredSize(int64_t* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT;

        Result GetMinimumProgramIndex(uint8_t* outValue, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result ListProgramIndex(int* outValue, uint8_t* outList, int numList, const ncm::ContentMetaKey& key, ncm::ContentType type, int offset) const NN_NOEXCEPT;

    public:
        static const int MaxStorageCount = 8;
        static const int MaxContentCount = 256;

    private:
        enum class SearchOption : Bit8
        {
            None,
            Exclude,
            Only
        };
        Result GetContentMetaStorageImpl(ncm::StorageId* outValue, const ncm::ContentMetaKey& key, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result GetContentIdImpl(ncm::ContentId* outValue, const ncm::ContentMetaKey& key, ncm::ContentType type, uint8_t idOffset, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result GetContentPathImpl(ncm::Path* outValue, ncm::ContentId contentId, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result GetContentIdFileSizeImpl(int64_t* pOutSize, ncm::ContentId contentId, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result ListContentInfoImpl(int* pOutCount, ncm::ContentInfo* pOutInfo, int numInfo, const ncm::ContentMetaKey& key, int offset, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result ReadContentIdFileImpl(void* buffer, size_t bufferSize, ncm::ContentId contentId, int64_t offset, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result WriteContentIdFileImpl(ncm::ContentId contentId, int64_t offset, const void* buffer,size_t bufferSize, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) NN_NOEXCEPT;
        Result GetContentIdFileHashImpl(ncm::Hash* pOutHash, ncm::ContentId contentId, ncm::ContentId metaContentId, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;

        struct ContentManagerUnit
        {
            ContentManagerUnit() NN_NOEXCEPT : storageId(ncm::StorageId::None), db(), storage(), needsCommit() {}
            void Initialize(ncm::StorageId sid, ncm::ContentMetaDatabase&& d, ncm::ContentStorage&& s) NN_NOEXCEPT
            {
                storageId = sid;
                db = std::move(d);
                storage = std::move(s);
            }

            void Reset() NN_NOEXCEPT
            {
                storageId = ncm::StorageId::None;
                db = ncm::ContentMetaDatabase();
                storage = ncm::ContentStorage();
            }

            bool IsRegistered() const NN_NOEXCEPT
            {
                return storageId != ncm::StorageId::None;
            }

            bool IsInstallableStorage() const NN_NOEXCEPT
            {
                return IsRegistered() && storageId != ncm::StorageId::Card;
            }

            bool IsTargetStorage(ncm::StorageId targetStorage) const NN_NOEXCEPT
            {
                return IsRegistered() && (targetStorage == ncm::StorageId::Any || targetStorage == storageId);
            }

            ncm::StorageId storageId;
            ncm::ContentMetaDatabase db;
            ncm::ContentStorage storage;
            bool needsCommit;
        };

        template <class T>
        bool IsSearchTarget(SearchOption option, const T a, const T b) const NN_NOEXCEPT;

        bool IsTargetStorage(const ContentManagerUnit& unit, SearchOption option, ncm::StorageId optoinTarget, ApplicationStoragePriority currentPriority) const NN_NOEXCEPT;
        Result CalculateStorage(int64_t* outValue, const ContentManagerUnit& unit, const ncm::ContentMetaKey& key) const NN_NOEXCEPT;
        Result ReadHash(ncm::Hash* outValue, ncm::ContentId contentId, const ContentManagerUnit& unit, ncm::ContentId metaContentId) const NN_NOEXCEPT;
        Result FindContentManagerUnitByKey(const ContentManagerUnit** outValue, const ncm::ContentMetaKey& key, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result FindContentManagerUnitById(const ContentManagerUnit** outValue, const ncm::ContentId& id, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;
        Result GetRequiredApplicationVersionImpl(uint32_t* outValue, const ncm::ContentMetaKey& key, SearchOption option = SearchOption::None, ncm::StorageId optionTarget = ncm::StorageId::None) const NN_NOEXCEPT;

        ContentManagerUnit m_List[MaxStorageCount];
        mutable NonRecursiveMutex m_Mutex;
        mutable ncm::ContentId m_ContentList[MaxContentCount];
        mutable bool m_OrphanList[MaxContentCount];
    };
}}}
