﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/arp/arp_Api.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/ns/srv/ns_ApplicationEntityManager.h>
#include <nn/ns/srv/ns_SystemReportManager.h>

#include <nn/ovln/ovln_SenderForOverlay.h>

namespace nn { namespace ns { namespace srv {

class CommitManager
{
public:
    enum class DynamicCommitStatus : Bit8
    {
        NoTarget,
        FullCommittable,
        PartialCommittable,
        RebootApplicationRequired,
        SystemUpdateRequired,
    };

    CommitManager() NN_NOEXCEPT : m_EnabledFlagMutex(false), m_CommitMutex(true), m_DynamicCommitEvent(os::EventClearMode_AutoClear, true) {}

    void Initialize(ApplicationRecordDatabase* recordDb, ApplicationEntityManager* entityManager, IntegratedContentManager* integrated, SystemReportManager* systemReportManager, ovln::SenderForOverlayType* sender) NN_NOEXCEPT;


    // Commit 呼び出し可能な状態かを取得する
    // 具体的には、ViewManager から呼ばれる想定
    Result IsDownloadTaskCommittable(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_RESULT_DO(IsDownloadTaskCommittable(outValue, id, false));
        NN_RESULT_SUCCESS;
    }
    Result IsApplyDeltaTaskCommittable(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    // Commit は、一時的に自動コミットを委ねられていると考えて、Commit を行う API
    // 具体的には、ViewManager から呼ばれる想定
    // RequestServer の制御は呼び出し元の責任
    Result CommitDownloadTask(ncm::ApplicationId id) NN_NOEXCEPT;
    Result CommitApplyDeltaTask(ncm::ApplicationId id) NN_NOEXCEPT;

    // ローカルコンテンツ配信用
    Result CommitReceiveTask(ncm::ApplicationId id) NN_NOEXCEPT;

    // Notify は、CommitManager に Commit するか委ねる API
    // 具体的には、ApplicationInstallRequest から呼ばれる想定
    // RequestServer の制御は呼び出し元の責任
    Result NotifyDownloadTaskCompleted(nim::NetworkInstallTaskId id) NN_NOEXCEPT;
    Result NotifyApplyDeltaTaskCompleted(nim::ApplyDeltaTaskId id) NN_NOEXCEPT;

    // アプリケーションが動いている状態でのコミットを試す API
    // 具体的には、アプリから呼ばれる想定
    // RequestServer の制御は呼び出し元の責任
    Result IsCurrentApplicationDownloadTaskDynamicallyCommittable(bool* outValue) NN_NOEXCEPT;
    Result CommitCurrentApplicationDownloadTaskDynamically() NN_NOEXCEPT;

    // 有効無効の制御。このレイヤでは複数プロセスから扱われるなどは制御しない
    void EnableAutoCommit() NN_NOEXCEPT;
    void DisableAutoCommit() NN_NOEXCEPT;

    // デバッグ用に、外部から dyanmic commit イベントを引く
    Result TriggerDynamicCommitEvent() NN_NOEXCEPT;

    // Dynamic Commit 制御用の情報を渡す
    void RegisterRuntimeInstallPolicy(ncm::ApplicationId id, RuntimeAddOnContentInstall policy) NN_NOEXCEPT
    {
        m_RegisteredId = id;
        m_RegisteredRuntimeInstallPolicy = policy;
    }

    // Dynamic Commit が行われたことを通知するイベントを返す
    os::SystemEvent& GetDynamicCommitEvent() NN_NOEXCEPT
    {
        // 本当は aocsrv でクリアするべきだが、実装が複雑になるだけなので今はここでクリアしてしまう
        m_DynamicCommitEvent.Clear();
        return m_DynamicCommitEvent;
    }

    // 保持している間は commit を遅延する lock を取得する
    std::unique_lock<os::Mutex> GetCommitStopper() NN_NOEXCEPT;

private:
    // 内部用の IsDownloadTaskCommittable
    Result IsDownloadTaskCommittable(bool* outValue, ncm::ApplicationId id, bool checkRequiredSystemVersion) NN_NOEXCEPT;
    Result CommitDownloadTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT;
    Result CommitAddOnContentOnlyImpl(ncm::ApplicationId id) NN_NOEXCEPT;
    Result CommitApplyDeltaTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT;
    Result CommitReceiveTaskImpl(ncm::ApplicationId id) NN_NOEXCEPT;
    Result ReportThroughput(ncm::ApplicationId id) NN_NOEXCEPT;
    Result SendDownloadCompleteOverlayNotification(ncm::ApplicationId id) NN_NOEXCEPT;

    Result IsDownloadTaskCommittableStrictly(bool* outValue, ncm::ApplicationId id) NN_NOEXCEPT;
    Result CheckDynamicCommitStatus(DynamicCommitStatus* outValue, const arp::ApplicationLaunchProperty& property) NN_NOEXCEPT;

    bool IsDynamicCommitRequested(ncm::ApplicationId id) NN_NOEXCEPT;
    bool IsAutoCommitEnabled() NN_NOEXCEPT;

    bool m_IsAutoCommitEnabled { false }; // 上位システムから、自動コミットをゆだねられているかどうか
    bool m_IsAutoCommitEnabledForcibly { false };  // fwdbg による強制自動化
    bool m_IsAutoCommitDisabledForcibly { false }; // fwdbg による強制無効化。無効化のほうが優先される
    os::Mutex m_EnabledFlagMutex;
    os::Mutex m_CommitMutex;

    ncm::ApplicationId m_RegisteredId {}; // Dynamic Commit 制御情報を渡された id
    RuntimeAddOnContentInstall m_RegisteredRuntimeInstallPolicy {RuntimeAddOnContentInstall::Deny }; // Dynamic Commit 処理種類
    os::SystemEvent m_DynamicCommitEvent;

    ApplicationRecordDatabase* m_RecordDatabase;
    ApplicationEntityManager* m_EntityManager;
    IntegratedContentManager* m_IntegratedManager;
    SystemReportManager* m_SystemReportManager;

    ovln::SenderForOverlayType* m_Sender;

    static NonRecursiveMutex s_KeyMutex;
    static ncm::StorageContentMetaKey s_KeyList[];

};

}}}
