﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Optional.h>
#include <nn/ns/srv/ns_ApplicationInstallRequest.h>
#include <nn/ns/srv/ns_BgtcTask.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/ns/srv/ns_VulnerabilityManager.h>
#include <nn/ns/ns_NotificationSystemApi.h>
#include <nn/util/util_TinyMt.h>

#define NN_NM_TRACE(...) NN_DETAIL_NS_TRACE("[AsyncTaskManager] " __VA_ARGS__)

namespace nn { namespace ns { namespace srv {

    typedef util::optional<AsyncTask> AsyncTaskOpt;

    class AsyncTaskManager
    {
        NN_DISALLOW_COPY(AsyncTaskManager);
        NN_DISALLOW_MOVE(AsyncTaskManager);
    public:
        static const int AsyncTaskListSize = static_cast<int>(AsyncTask::AsyncTaskType::NumberOfAsyncTaskType);
        typedef std::array<AsyncTaskOpt, AsyncTaskListSize> AsyncTaskList;

        struct Config
        {
            bool    isNetworkUpdateEnabled;
            bool    isDownloadTaskListEnabled;
            bool    isVersionListEnabled;
            bool    isRandomWaitEnabled;
            bool    isETicketAvailableEnabled;
            bool    isSendRightsUsageStatusRequestEnabled;
            bool    isSyncELicenseRequestEnabled;
            uint32_t debugWaitingLimit;
            uint32_t versionListWaitingLimit;
            bool    useVersionListWaitPolicyForTest;
        };

        struct NotificationHandlerHolder
        {
            std::function<nn::Result(const NotificationInfo& notificationInfo)> handler;
            AsyncTask::AsyncTaskType notificationType;
            NotificationHandlerHolder* next;
        };

        AsyncTaskManager(uint8_t* taskThreadStack, size_t taskThreadStackSize) NN_NOEXCEPT
            :   m_Config({}), m_RequestList(nullptr), m_RequestServer(nullptr),
                m_TaskThread(), m_StopEvent(os::EventClearMode_AutoClear, false), m_HandlerHolderRoot({}),
                m_AsyncTaskList(), m_TaskThreadStack(taskThreadStack), m_TaskThreadStackSize(taskThreadStackSize), m_Mt({}),
                m_TaskThreadLock(false), m_TaskThreadRunning(false), m_TaskListLock(false), m_AutoUpdateTask({AsyncTask::AsyncTaskType::AutoUpdate})
        {}
        ~AsyncTaskManager() NN_NOEXCEPT
        {
            Finalize();
        }

        Result Initialize(ApplicationInstallRequestList* requestList, RequestServer* requestServer, ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT;
        Result Initialize(
            NotificationHandlerHolder* holders,
            int holderCount,
            const Config& config,
            ApplicationInstallRequestList* requestList,
            RequestServer* requestServer,
            ApplicationVersionManager* applicationVersionManager) NN_NOEXCEPT;

        Result Register(const NotificationInfo& info) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT
        {
            Stop();
            if (m_BgTask.IsInitialized())
            {
                m_BgTask.Finalize();
            }
        }
        Result RequestDownloadTaskList(const nim::ETag& etag, bool notifiesRequiredSystemUpdate) NN_NOEXCEPT;
        Result RequestVersionList(const nim::ETag& etag) NN_NOEXCEPT;
        Result RequestDownloadTicket(uint64_t titleId) NN_NOEXCEPT;
        Result ListNotificationTask(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ns::AsyncTask>& outList) NN_NOEXCEPT;

    private:


        void TaskThreadFunc() NN_NOEXCEPT;
        void Start() NN_NOEXCEPT;
        void Stop() NN_NOEXCEPT;
        void AddHandler(NotificationHandlerHolder* holder) NN_NOEXCEPT;
        Config MakeDefaultConfig() NN_NOEXCEPT;
        void MakeAsyncTask(AsyncTask* out, const NotificationInfo& info) NN_NOEXCEPT;
        int64_t CalcWaitingLimitFromWaitingPolicy(const NotificationInfo::WaitingPolicy& waitingPolicy) NN_NOEXCEPT;
        int64_t CalcVersionListWaitingLimit() NN_NOEXCEPT;
        void RegisterTaskImpl(const AsyncTask& task) NN_NOEXCEPT;
        void UpdateAutoUpdateTask(const NotificationInfo& info) NN_NOEXCEPT;

        Config m_Config;
        ApplicationInstallRequestList* m_RequestList;
        RequestServer* m_RequestServer;
        ApplicationVersionManager* m_ApplicationVersionManager;

        os::ThreadType m_TaskThread;
        os::SystemEvent m_StopEvent;

        NotificationHandlerHolder m_HandlerHolderRoot;
        AsyncTaskList m_AsyncTaskList;

        uint8_t* m_TaskThreadStack;
        size_t m_TaskThreadStackSize;
        util::TinyMt m_Mt;

        os::Mutex m_TaskThreadLock;
        bool m_TaskThreadRunning;

        BgtcTask m_BgTask;

        os::Mutex m_TaskListLock;

        AsyncTask m_AutoUpdateTask;
    };
}}}
