﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/ns/srv/detail/json/ns_ResultForJson.h>

namespace nn { namespace ns { namespace srv { namespace detail { namespace json {

// 外部からキャンセル可能な概念
class Cancellable
{
private:
    std::atomic<bool> m_UserCancel;

protected:
    Cancellable() NN_NOEXCEPT;

public:
    void Cancel() NN_NOEXCEPT;
    bool IsCancelled() const NN_NOEXCEPT;
};

// Cancellable を注入可能な概念
// (と言っても、キャンセルする/しないは実装依存)
class CancelInjectable
{
private:
    const Cancellable* const m_pCancellable;

protected:
    explicit CancelInjectable(const Cancellable* pCancellable) NN_NOEXCEPT;
    bool IsCancelled() const NN_NOEXCEPT;
};


}}}}} // ~namespace nn::ns::srv::detail::json
/* --------------------------------------------------------------------------------------------
    実装
 */
#include <mutex>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ns { namespace srv { namespace detail { namespace json {

/* ------------------------------------------------------------
    Cancellable & CancelInjectable
 */
inline Cancellable::Cancellable() NN_NOEXCEPT
    : m_UserCancel(false)
{
}
inline void Cancellable::Cancel() NN_NOEXCEPT
{
    auto before = m_UserCancel.exchange(true);
    NN_SDK_ASSERT(!before);
    NN_UNUSED(before);
}
inline bool Cancellable::IsCancelled() const NN_NOEXCEPT
{
    return m_UserCancel.load();
}
inline CancelInjectable::CancelInjectable(const Cancellable* pCancellable) NN_NOEXCEPT
    : m_pCancellable(pCancellable)
{
}
inline bool CancelInjectable::IsCancelled() const NN_NOEXCEPT
{
    return m_pCancellable && m_pCancellable->IsCancelled();
}

}}}}} // ~namespace nn::ns::srv::detail::json
