﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/ns/ns_ApplicationRightsTypes.h>
#include <nn/ns/ns_Async.h>

namespace nn { namespace ns {

    typedef AsyncValueList<ApplicationRightsOnServer> AsyncApplicationRightsOnServerList;

    /**
    * @brief    アプリケーションの利用権利情報を取得します。
    *
    * @details  サーバーにアプリケーションの利用権利情報を問い合わせます。
    *           結果は AsyncApplicationRightsOnServerList.Get() を用いて取得してください。
    *           AsyncApplicationRightsOnServerList.Get() でアプリケーションと追加コンテンツの ApplicationRightsOnServer が一括して取得できます。
    *           指定した Uid に NintendoAccount が連携されていない場合は、機器認証の権利のみを問い合わせます。
    *
    * @post
    *           - AsyncApplicationRightsOnServerList が利用できるようになります。
    *
    * @retval   処理の結果が返ります。
    * @retval   ResultOutOfMaxRunningTask               同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultApplicationRecordNotFound         アプリケーション記録が見つかりません。
    * @retval   ResultApplicationContentNotFound        (AsyncApplicationRightsOnServer から返される Result) 必要なコンテンツが見つかりません。
    * @retval   ResultApplicationRightsAlreadyExists    (AsyncApplicationRightsOnServer から返される Result) 既に必要な権限が揃っています
    *
    */
    Result RequestApplicationRightsOnServer(
        AsyncApplicationRightsOnServerList* pOutValue,
        ncm::ApplicationId id,
        const account::Uid& uid) NN_NOEXCEPT;

    /**
    * @brief    アプリケーションの利用権利情報を取得します。
    *
    * @details  サーバーにアプリケーションの機器認証の利用権利情報を問い合わせます。
    *           結果は AsyncApplicationRightsOnServerList.Get() を用いて取得してください。
    *           AsyncApplicationRightsOnServerList.Get() でアプリケーションと追加コンテンツの ApplicationRightsOnServer が一括して取得できます。
    *
    * @post
    *           - AsyncApplicationRightsOnServerList が利用できるようになります。
    *
    * @retval   処理の結果が返ります。
    * @retval   ResultOutOfMaxRunningTask               同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultApplicationRecordNotFound         アプリケーション記録が見つかりません。
    * @retval   ResultApplicationContentNotFound        (AsyncApplicationRightsOnServer から返される Result) 必要なコンテンツが見つかりません。
    * @retval   ResultApplicationRightsAlreadyExists    (AsyncApplicationRightsOnServer から返される Result) 既に必要な権限が揃っています
    *
    */
    Result RequestApplicationRightsOnServer(
        AsyncApplicationRightsOnServerList* pOutValue,
        ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション、および、追加コンテンツの権利要求を行います。
    *
    * @details  サーバー上で権利割り当てを行った後、権利をダウンロードします。
    *           動的権利の場合、ダウンロードしただけでは使うことが出来ません。
    *           RegisterRestrictiveRightsUser() を呼ぶことで利用できるようになります。
    *
    * @post
    *           - AsyncResult が利用できるようになります。
    *
    * @retval   処理の結果が返ります。
    * @retval   ResultOutOfMaxRunningTask               同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultApplicationRecordNotFound         アプリケーション記録が見つかりません。
    */
    Result RequestAssignRights(
        AsyncResult* pOutValue,
        const ApplicationRightsOnServer* pList,
        int listCount) NN_NOEXCEPT;

    /**
    * @brief    現在利用中の権利の再割り当てを行ないます
    *
    * @details  サーバへの問い合わせ、サーバー上での権利割り当て、
    *           そしてチケットのダウンロードを行ないます。
    *
    * @post
    *           - AsyncResult が利用できるようになります。
    *
    * @retval   処理の結果が返ります。
    * @retval   ResultOutOfMaxRunningTask               同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultRequiredSystemUpdate              本体更新が必要です。
    */
    Result RequestAssignRightsToResume(
        AsyncResult* pOutValue,
        RightsEnvironmentHandle handle,
        const account::Uid& uid) NN_NOEXCEPT;

    /**
    * @brief    動的権利の利用者を登録します。
    *
    * @details  本体上の動的権利の利用者を登録します。
    *           アプリケーションの起動のたびに呼び出す必要があります。
    *
    * @retval   処理の結果が返ります。
    */
    Result RegisterUserOfAccountRestrictedRights(const account::Uid& uid) NN_NOEXCEPT;

    /**
    * @brief    動的権利の利用者の登録情報を削除します。
    *
    * @retval   処理の結果が返ります。
    */
    void UnregisterAllUsersOfAccountRestrictiveRights() NN_NOEXCEPT;

    /**
     * @brief   現在有効な権利の所有者の状態を検証します。
     *
     * @retval  処理の結果が返ります。
     * @retval  ResultInactiveNintendoAccount           アプリケーションを起動する権利を持っている NA が無効状態です。
     */
    Result VerifyActivatedRightsOwners(RightsEnvironmentHandle handle) NN_NOEXCEPT;

    /**
    * @brief    権利の失効理由が有効期限切れかどうかを判定します
    * @brief    利用している権利の状態を取得します。
    *
    * @pre
    *           - アプリケーションが起動している。
    *
    * @details  現在利用中の権利の状態を取得します。
    *           権利を喪失したという通知を受けた後に、当該 API を呼び出したとしても、
    *           この API の呼び出しの直前に権利の延長が成功し、利用可能になっていることがあります。
    *
    * @retval   利用している権利の状態が返ります。
    */
    ApplicationRightsStatus GetApplicationRightsStatus(RightsEnvironmentHandle handle) NN_NOEXCEPT;

}}  // namespace nn::ns
