﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/ns/ns_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns {

namespace detail{
    class IAsyncValue;
    class IAsyncResult;
    class IProgressAsyncResult;

    template<typename T>
    class AsyncBase
    {
    public:
        ~AsyncBase() NN_NOEXCEPT;
        void Initialize(sf::SharedPointer<T> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;
        void Wait() NN_NOEXCEPT;
        bool TryWait() NN_NOEXCEPT;
        void Cancel() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        os::SystemEvent& GetEvent() NN_NOEXCEPT;
        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    protected:
        sf::SharedPointer<T> m_Async;

    private:
        util::optional<os::SystemEvent> m_Event;
    };

    class AsyncValueImpl : private detail::AsyncBase<detail::IAsyncValue>
    {
    public:
        ~AsyncValueImpl() NN_NOEXCEPT;
        void Initialize(sf::SharedPointer<detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;
        Result Get(void* buffer, size_t bufferSize) NN_NOEXCEPT;
        uint64_t GetSize() NN_NOEXCEPT;
        void Wait() NN_NOEXCEPT;
        bool TryWait() NN_NOEXCEPT;
        void Cancel() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        os::SystemEvent& GetEvent() NN_NOEXCEPT;
        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    private:
        typedef detail::AsyncBase<detail::IAsyncValue> BaseType;
    };

    class ProgressAsyncResultImpl : private detail::AsyncBase<detail::IProgressAsyncResult>
    {
    public:
        ~ProgressAsyncResultImpl() NN_NOEXCEPT;
        void Initialize(sf::SharedPointer<detail::IProgressAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;
        Result Get() NN_NOEXCEPT;
        void Wait() NN_NOEXCEPT;
        bool TryWait() NN_NOEXCEPT;
        void Cancel() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        os::SystemEvent& GetEvent() NN_NOEXCEPT;
        void GetProgress(void* buffer, size_t bufferSize) NN_NOEXCEPT;
        Result GetDetailResult() NN_NOEXCEPT;
        void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    private:
        typedef detail::AsyncBase<detail::IProgressAsyncResult> BaseType;
    };
}

class AsyncResult : private detail::AsyncBase<detail::IAsyncResult>
{
public:

    /**
    * @brief    非同期 Result の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<detail::IAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT;

    /**
    * @brief    非同期 Result のデストラクタです。初期化済みのインスタンスには Finalize と同等の処理が行われます。
    */
    ~AsyncResult() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get() NN_NOEXCEPT;

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT;

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。
    */
    void Finalize() NN_NOEXCEPT;

    os::SystemEvent& GetEvent() NN_NOEXCEPT;

private:
    typedef detail::AsyncBase<detail::IAsyncResult> BaseType;
};

template<typename T>
class AsyncValue : private detail::AsyncValueImpl
{
public:
    typedef T AsyncValueType;

    /**
    * @brief    非同期 Value の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Initialize(async, nativeHandle);
    }

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get(T* outValue) NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::Get(outValue, sizeof(T));
    }

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::GetErrorContext(outValue);
    }

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Wait();
    }

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::TryWait();
    }

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Cancel();
    }

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。デストラクタでも同様の処理が行われます。
    */
    void Finalize() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Finalize();
    }

    os::SystemEvent& GetEvent() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::GetEvent();
    }
};

template<typename T>
class AsyncValueList : private detail::AsyncValueImpl
{
public:
    typedef T AsyncValueType;

    /**
    * @brief    非同期 Value の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<detail::IAsyncValue> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Initialize(async, nativeHandle);
    }

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get(int* outCount, T* outList, int count) NN_NOEXCEPT
    {
        Wait();
        uint64_t size = detail::AsyncValueImpl::GetSize();
        NN_RESULT_THROW_UNLESS(size <= static_cast<uint64_t>(sizeof(T) * count), ResultBufferNotEnough());
        NN_RESULT_DO(detail::AsyncValueImpl::Get(outList, sizeof(T) * count));
        *outCount = static_cast<int>(size / sizeof(T));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        detail::AsyncValueImpl::GetErrorContext(outValue);
    }

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Wait();
    }

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::TryWait();
    }

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Cancel();
    }

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。デストラクタでも同様の処理が行われます。
    */
    void Finalize() NN_NOEXCEPT
    {
        detail::AsyncValueImpl::Finalize();
    }

    os::SystemEvent& GetEvent() NN_NOEXCEPT
    {
        return detail::AsyncValueImpl::GetEvent();
    }
};

template<typename T>
class ProgressAsyncResult : private detail::ProgressAsyncResultImpl
{
public:
    /**
    * @brief    非同期 Result の初期化を行います。通常は他の関数によって初期化されるため、呼び出す必要はありません。
    */
    void Initialize(sf::SharedPointer<detail::IProgressAsyncResult> async, const sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::Initialize(async, nativeHandle);
    }

    /**
    * @brief    対象の非同期処理を Wait してから結果を取得します。
    */
    Result Get() NN_NOEXCEPT
    {
        return detail::ProgressAsyncResultImpl::Get();
    }

    /**
    * @brief    非同期処理失敗時のエラー文脈を取得します。
    */
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::GetErrorContext(outValue);
    }

    /**
    * @brief    対象の非同期処理を Wait します。
    */
    void Wait() NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::Wait();
    }

    /**
    * @brief    対象の非同期処理を TryWait します。
    */
    bool TryWait() NN_NOEXCEPT
    {
        return detail::ProgressAsyncResultImpl::TryWait();
    }

    /**
    * @brief    対象の非同期処理を Cancel します。Wait しているスレッドはシグナルされます。
    */
    void Cancel() NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::Cancel();
    }

    /**
    * @brief    対象の非同期処理に必要なリソースを開放して、非同期処理を破棄します。
    */
    void Finalize() NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::Finalize();
    }

    /**
    * @breif    現在の進捗を取得します。
    */
    void GetProgress(T* outValue) NN_NOEXCEPT
    {
        detail::ProgressAsyncResultImpl::GetProgress(outValue, sizeof(T));
    }

    /**
    * @breif    詳細な Result を取得します。
    */
    Result GetDetailResult() NN_NOEXCEPT
    {
        return detail::ProgressAsyncResultImpl::GetDetailResult();
    }

    os::SystemEvent& GetEvent() NN_NOEXCEPT
    {
        return detail::ProgressAsyncResultImpl::GetEvent();
    }
};

}}

