﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentType.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace ns {
    struct ApplicationRightsOnServerFlagTag {};
    typedef util::BitFlagSet<32, ApplicationRightsOnServerFlagTag> ApplicationRightsOnServerFlag;
    struct ApplicationRightsUnavailableFlagTag {};
    typedef util::BitFlagSet<32, ApplicationRightsUnavailableFlagTag> ApplicationRightsUnavailableFlag;
    struct ApplicationRightsOnClientFlagTag {};
    typedef util::BitFlagSet<32, ApplicationRightsOnClientFlagTag> ApplicationRightsOnClientFlag;

    enum class ApplicationContentType : Bit8
    {
        Unknown,
        Application,
        AddOnContent,
    };

    struct ApplicationRightsUnavailableReason
    {
        ApplicationRightsUnavailableFlag _flags; //!< ライブラリ内部で利用します
        Bit8 reserved[4];

        /**
         *  @brief  権利がありません
         */
        bool NoRights() const NN_NOEXCEPT;

        /**
         *  @brief  本体更新が必要です
         */
        bool SystemUpdateRequired() const NN_NOEXCEPT;

        /**
         *  @brief  機器認証以外の権利を発行できません
         */
        bool HasDeviceLinkedRightsOnlyContent() const NN_NOEXCEPT;

        /**
         *  @brief  配信前です。
         */
        bool NotReleased() const NN_NOEXCEPT;

        /**
         *  @brief  付与上限数に達しています
         */
        bool AssignableRightsLimitExceeded() const NN_NOEXCEPT;
    };

    struct ApplicationRightsOnServer
    {
        ncm::ApplicationId id; //!< アプリケーションの ID です。
        account::Uid uid; //!< アカウントの UID です。
        ApplicationRightsOnServerFlag _flags; //!< ライブラリ内部で利用します
        ApplicationContentType type; //!< コンテンツの種類です
        Bit8 _handle; //!< ライブラリ内部で利用します
        Bit8 reserved0x18[2];
        ApplicationRightsUnavailableReason reason; //!< 利用不可の場合の理由です

        /**
         *  @brief  サーバー上に利用可能な権利があります。
         */
        bool HasAvailableRights() const NN_NOEXCEPT;

        /**
         *  @brief  利用不可な権利があります。
         */
        bool HasUnavailableRights() const NN_NOEXCEPT;

        /**
        *   @brief    アカウントに限定された権利です。
        */
        bool IsAccountRestrictedRights() const NN_NOEXCEPT;

        /**
        *   @brief    チケット同期で権利を取得できます。
        */
        bool RecommendSyncTicket() const NN_NOEXCEPT;

        /**
        *   @brief    権利要求で権利を取得できます。
        */
        bool RecommendAssignRights() const NN_NOEXCEPT;

    };
    NN_STATIC_ASSERT(sizeof(ApplicationRightsOnServer) == 40);

    struct ApplicationRightsOnClient
    {
        ncm::ApplicationId id; //!< アプリケーションの ID です。
        account::Uid uid; //!< アカウントの UID です。未指定の場合は account::InvalidUid が入ります
        ApplicationRightsOnClientFlag _flags; //!< ライブラリ内部で利用します
        ApplicationContentType type; //!< コンテンツの種類です
        Bit8 reserved[3];

        /**
         *  @brief      利用可能な権利があります
         */
        bool HasAvailableRights() const NN_NOEXCEPT;

        /**
         *  @brief      利用不可な権利があります。
         */
        bool HasUnavailableRights() const NN_NOEXCEPT;

        /**
        *   @brief      アカウントに限定して利用できる権利があります。
        *
        *   @details    権利の利用者に uid が指し示すアカウントを設定した時に有効になる権利がある場合に True になります。
        *               GetApplicationRightsOnClient() に uid を指定しなかった場合は常に False になります。
        */
        bool HasAccountRestrictedRights() const NN_NOEXCEPT;

        /**
        *   @brief      権利状態をサーバーに問い合わせることを推奨します
        */
        bool RecommendInquireServer() const NN_NOEXCEPT;

        /**
        *   @brief      予約購入の権利があります。
        *
        *   @details    HasUnavailableRights() が True の時にこのメソッドの結果が意味を持ちます
        */
        bool HasPrepurchasedRights() const NN_NOEXCEPT;

    };
    NN_STATIC_ASSERT(sizeof(ApplicationRightsOnClient) == 32);

    enum class RightsEnvironmentHandle : Bit64
    {
    };

    struct RightsEnvironmentStatus
    {
        bool isAvailable;
        Bit8 _reserved[7];
        int64_t expirationTimeInTick;
    };

    enum class ApplicationRightsStatus : Bit8
    {
        Available,      //!< 権利が利用可能です。
        Expired,        //!< 有効期限が失効しています。
        Unavailable,    //!< 権利を利用出来ません。
    };

}}  // namespace nn::ns
