﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ncm/ncm_ContentId.h>
#include <nn/os/os_Mutex.h>

namespace nn { namespace nim { namespace srv {

// 現状エンディアンが異なる通信はサポートしておらず、リトルエンディアン前提のプロトコルとなっているので、assert しておく
NN_STATIC_ASSERT(NN_BUILD_CONFIG_ENDIAN_LITTLE);

enum LocalCommunicationDeliveryProtocolTag : Bit8
{
    EndSessionTag,
    GetPackagedContentInfoTag,
    GetContentTag,
    GetCommonTicketTag,
    SendTotalSizeTag,
};

struct LocalCommunicationDeliveryProtocolHeader
{
    char                                  signature[4];
    LocalCommunicationDeliveryProtocolTag tag;
    Bit8                                  reserved0x5[1];
    uint16_t                              additionalHeaderSize;
    int64_t                               size;
};
NN_STATIC_ASSERT(sizeof(LocalCommunicationDeliveryProtocolHeader) == 0x10);

struct LocalCommunicationContentInfo
{
    ncm::ContentId contentId;
    bool isTemporary;
    Bit8 reserved[7];
};
NN_STATIC_ASSERT(sizeof(LocalCommunicationContentInfo) == 0x18);

struct LocalCommunicationDeliveryProtocolGetCommonTicketResponseHeader
{
    uint32_t ticketSize;
    uint32_t certificateSize;
};

class LocalCommunicationDeliveryProtocolBase
{
public:
    LocalCommunicationDeliveryProtocolBase() NN_NOEXCEPT : m_CancelMutex(false) {}

    typedef std::function<Result(size_t, int64_t)> UpdateSendBufferFunction;
    typedef std::function<Result(size_t, int64_t)> ProcessReceiveBufferFunction;

    Result Send(const LocalCommunicationDeliveryProtocolHeader& header, const void* buffer, size_t size, UpdateSendBufferFunction updateFunc = nullptr) NN_NOEXCEPT;
    Result SendHeader(const LocalCommunicationDeliveryProtocolHeader& header) NN_NOEXCEPT;
    Result SendData(const void* buffer, size_t size, int64_t contentLength, UpdateSendBufferFunction updateFunc = nullptr) NN_NOEXCEPT;
    Result ReceiveHeader(LocalCommunicationDeliveryProtocolHeader* outVe) NN_NOEXCEPT;
    Result ReceiveData(void* buffer, size_t bufferSize, int64_t contentLength, ProcessReceiveBufferFunction recvFunc = nullptr) NN_NOEXCEPT;

    void Cancel() NN_NOEXCEPT;
    void ResetCancel() NN_NOEXCEPT;
    bool IsCancelRequested() const NN_NOEXCEPT;

protected:
    int m_Socket { -1 };
    mutable os::Mutex m_CancelMutex;
    bool m_CancelRequest {};

    Result WaitClientConnection(int socketFd) NN_NOEXCEPT;
    Result HandleSocketReturnCode(int returnCode) NN_NOEXCEPT;
    Result SetLinger(int socketFd) NN_NOEXCEPT;
};

// 1 対 1 しか考慮していない
class LocalCommunicationDeliveryProtocolServer : public LocalCommunicationDeliveryProtocolBase
{
public:
    ~LocalCommunicationDeliveryProtocolServer() NN_NOEXCEPT;
    Result Initialize(uint32_t ipv4, uint16_t port) NN_NOEXCEPT;
    Result WaitClient() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
private:
    // TORIAEZU
    int m_ServerSocket { -1 };
    uint32_t m_PeerAddress;
};

class LocalCommunicationDeliveryProtocolClient : public LocalCommunicationDeliveryProtocolBase
{
public:
    ~LocalCommunicationDeliveryProtocolClient() NN_NOEXCEPT;
    Result Initialize(uint32_t ipv4, uint16_t port) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
};

LocalCommunicationDeliveryProtocolHeader MakeLocalCommunicationDeliveryProtocolRequestHeader(LocalCommunicationDeliveryProtocolTag tag, int64_t size);
LocalCommunicationDeliveryProtocolHeader MakeLocalCommunicationDeliveryProtocolResponseHeader(LocalCommunicationDeliveryProtocolTag tag, int64_t size);

Result CheckLocalCommunicationDeliveryProtocolSignature(const LocalCommunicationDeliveryProtocolHeader& header) NN_NOEXCEPT;

}}} // nn::ns::srv
