﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/es/es_Types.h>
#include <nn/ec/system/ec_DeviceAccountTypes.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/nim/nim_EciAccessorDefinition.h>
#include <nn/nim/srv/nim_HttpConnection.h>

namespace nn { namespace nim { namespace srv {

    class EciAccessor
    {
    public:
        EciAccessor() {};
        EciAccessor(DeviceContext* deviceContext, HttpConnection* connection) : m_Connection(connection)
        {
            m_DeviceId = deviceContext->GetDeviceId();
        };
        ~EciAccessor() {};

        Result Initialize(DeviceContext* deviceContext, HttpConnection* connection) NN_NOEXCEPT;

        // 各 REST API へのアクセス関数
        Result GetAccountStatus(ec::system::DeviceAccountStatus* outValue, const DeviceAccountToken& token) NN_NOEXCEPT;
        Result Register(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT;
        Result Unregister() NN_NOEXCEPT;
        Result AccountListETicketIds(char buffer[], size_t bufferSize, int* outTicketIdListCount, Bit64 outTicketIdList[], int outTicketIdListLength, const DeviceAccountId& id, const DeviceAccountToken& token) NN_NOEXCEPT;
        Result AccountListETicketIdsByRightsId(int* outTicketIdListCount, Bit64 outTicketIdList[], int outTicketIdListLength, const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
        Result AccountGetETickets(bool* outIsPrepurchaseRecordDownloaded, int* outComeBackAfter, char buffer[], size_t bufferSize, const DeviceAccountId& id, const DeviceAccountToken& token, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT;
        Result AccountGetETicketsForSyncTicket(char buffer[], size_t bufferSize, const DeviceAccountId& id, const DeviceAccountToken& token, const Bit64 ticketIdList[], int ticketIdListLength) NN_NOEXCEPT;
        Result GetRegistrationInfo(ec::system::DeviceAccountStatus* outStatus, DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT;
        Result CompleteETicketSync(const DeviceAccountToken& token) NN_NOEXCEPT;
        Result AccountTransfer(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT;
        Result SyncRegistration(DeviceAccountId* outAccountId, DeviceAccountToken* outDeviceToken) NN_NOEXCEPT;

        // ユーティリティ関数
        Result DownloadTicket(const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
        Result DownloadTicketForPrepurchasedContents(ec::system::TicketDownloadStatusForPrepurchasedContents* outStatus, const DeviceAccountId& id, const DeviceAccountToken& token, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;

        // for DynamicRights.
        //! @brief      柔軟な権利管理、動的チケットトークン取得用コンテキスト。
        //! @details    ワークバッファの配置は先頭から [ETicketTokenResponse], [EciPostData], [EciResponse] の順で利用します。
        class DynamicETicketsContext
        {
            NN_DISALLOW_COPY(DynamicETicketsContext);
            NN_DISALLOW_MOVE(DynamicETicketsContext);

        public:
            //! @brief      コンストラクタ。
            DynamicETicketsContext() NN_NOEXCEPT;

            //! @brief      コンテキストを初期化します。
            //!
            //! @param[in]  pBuffer     要求処理のワークとして利用可能な連続領域の先頭アドレス。@n
            //!                         開始アドレスは 4 byte アラインされている必要があります。
            //! @param[in]  bufferSize  要求処理のワークとして利用可能な連続領域の容量( byte 単位 )。
            //!
            //! @return     指定サイズのワーク領域で要求可能なライセンス数を返します。@n
            //!             0 の場合は、初期化に失敗した事を示します。
            //! @retval     0       @a requiredSize で指定された容量が最小要件を満たしませんでした。
            //! @retval     > 0     @a requiredSize で指定された容量で要求可能なライセンス数。
            //!
            //! @details    Dragons の "/v1/elicenses/eticket_token" のレスポンスバッファも考慮します。
            size_t Initialize(char* pBuffer, size_t bufferSize) NN_NOEXCEPT;

            //! @brief      ECI要求時のポストデータ作成に利用可能なワークバッファ中の領域先頭を取得します。
            //! @return     ポストデータ領域の先頭アドレスを返します。@n
            //!             nullptr の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            char* GetBufferForEciPostData() NN_NOEXCEPT;

            //! @brief      ECI要求時のレスポンス領域に利用可能なワークバッファ中の領域先頭を取得します。
            //! @return     レスポンスデータ領域の先頭アドレスを返します。@n
            //!             nullptr の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            char* GetBufferForEciResponse() NN_NOEXCEPT;

            //! @brief      動的チケットトークン要求レスポンスの解析に利用可能なワークバッファ中の領域先頭を取得します。
            //! @return     レスポンスデータ領域の先頭アドレスを返します。@n
            //!             nullptr の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            char* GetBufferForETicketToken() NN_NOEXCEPT;

            //! @brief      ECI要求時のポストデータ作成に利用可能なワークバッファ中の利用可能サイズを取得します。
            //! @return     ポストデータ領域の利用可能サイズを返します。@n
            //!             0 の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            NN_FORCEINLINE size_t GetCapacityForEciPostData() const NN_NOEXCEPT
            {
                return m_CapacityForEciPostData;
            }

            //! @brief      ECI要求時のレスポンス領域に利用可能なワークバッファ中の利用可能サイズを取得します。
            //! @return     ポストデータ領域の利用可能サイズを返します。@n
            //!             0 の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            NN_FORCEINLINE size_t GetCapacityForEciResponse() const NN_NOEXCEPT
            {
                return m_CapacityForEciResponse;
            }

            //! @brief      動的チケットトークン要求レスポンスの解析に利用可能なワークバッファ中の利用可能サイズを取得します。
            //! @return     ポストデータ領域の利用可能サイズを返します。@n
            //!             0 の場合、正しくコンテキストが準備されていません。
            //! @details    @ref Initialize() を実施している必要があります。
            NN_FORCEINLINE size_t GetCapacityForETicketToken() const NN_NOEXCEPT
            {
                return m_CapacityForTokenResponse;
            }

            //! @brief      コンテキスト初期化時に求めた要求可能なライセンス数を返します。
            //! @details    @ref Initialize() を実施している必要があります。
            NN_FORCEINLINE size_t GetAvailableLicenseCount() const NN_NOEXCEPT
            {
                return m_AvailableLicenseCount;
            }

        private:
            //! @brief      指定サイズのワーク領域で要求可能なライセンス数の算出。
            //!
            //! @param[in]  requiredSize    要求処理のワークとして利用可能な連続領域の容量( byte 単位 )。
            //!
            //! @return     指定サイズのワーク領域で要求可能なライセンス数を返します。
            //! @retval     0       @a requiredSize で指定された容量が最小要件を満たしませんでした。
            //! @retval     > 0     @a requiredSize で指定された容量で要求可能なライセンス数。
            //!
            //! @details    Dragons の "/v1/elicenses/eticket_token" のレスポンスバッファも考慮します。
            size_t CalculateAvailableLicenseCount(size_t requiredSize) NN_NOEXCEPT;

            //! @brief      コンテキストの利用可用性を確認します。
            NN_FORCEINLINE bool IsAvailable() const NN_NOEXCEPT
            {
                return (m_AvailableLicenseCount > 0u && nullptr != m_pBuffer && m_BufferSize > 0u);
            }

            char*   m_pBuffer;
            size_t  m_BufferSize;
            size_t  m_AvailableLicenseCount;
            size_t  m_CapacityForEciPostData;
            size_t  m_CapacityForEciResponse;
            size_t  m_CapacityForTokenResponse;
        };

        //! @brief      動的チケットダウンロード・インポート実施。
        //! @param[in]  context     動的チケット要求のためのコンテキスト。
        Result RequestDynamicETickets(DynamicETicketsContext& context, const DeviceAccountId& deviceAccountId, const DeviceAccountToken& deviceToken, const char* pDynamicETicketToken) NN_NOEXCEPT;

    private:
        // for DynamicRights.
        Result RequestDynamicETicketsImpl(char* pWorkBuffer, size_t bufferSize, char* pPostData) NN_NOEXCEPT;

    private:
        Bit64 m_DeviceId;
        HttpConnection* m_Connection;
    };

}}}
