﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

#include <nn/fs/fs_File.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>
#include <nn/os/os_Tick.h>

#include <nn/nim/nim_ApplyDownloadedDeltaProgress.h>

namespace nn { namespace nim { namespace srv {

class FileApplyDownloadedDeltaTaskMeta
{
public:
    enum class MetaVersion : Bit8
    {
        Version0 = 0, // 2NUP まで
        Version1 = 1  // 3NUP 以降
    };

public:
    FileApplyDownloadedDeltaTaskMeta() NN_NOEXCEPT {}
    ~FileApplyDownloadedDeltaTaskMeta() NN_NOEXCEPT;

    // 2 系までのフォーマット
    static Result Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, const ncm::ContentMetaKey* keyList, int count, ncm::StorageId storageId) NN_NOEXCEPT;

    // 3 系以降のフォーマット
    static Result Create(const char* path, ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, ncm::StorageId storageId, const ncm::StorageContentMetaKey* keyList, int count) NN_NOEXCEPT;

    Result Initialize(const char* path) NN_NOEXCEPT;
    int CountContentMeta() const NN_NOEXCEPT
    {
        return m_Header.contentMetaCount;
    }
    Result GetContentMeta(ncm::StorageContentMetaKey* outValue, int index) NN_NOEXCEPT;
    ncm::ApplicationId GetApplicationId() const NN_NOEXCEPT
    {
        return m_Header.applicationId;
    }
    ncm::ContentMetaKey GetSourceKey() const NN_NOEXCEPT
    {
        return m_Header.originalSource;
    }
    ncm::StorageId GetStorageId() const NN_NOEXCEPT
    {
        return m_Header.storageId;
    }
    MetaVersion GetMetaVersion() const NN_NOEXCEPT
    {
        return m_Header.version;
    }

private:
    struct Header
    {
        ncm::ApplicationId  applicationId;
        ncm::ContentMetaKey originalSource; // 初期に指定した更新対象
        int                 contentMetaCount;
        ncm::StorageId      storageId;
        MetaVersion         version;
        Bit8                reserved[2];
    };

    static Header MakeHeader(ncm::ApplicationId appId, const ncm::ContentMetaKey& sourceKey, int contentMetaCount, ncm::StorageId storageId, MetaVersion version) NN_NOEXCEPT
    {
        Header header = {};
        header.applicationId = appId;
        header.originalSource = sourceKey;
        header.contentMetaCount = contentMetaCount;
        header.storageId = storageId;
        header.version = version;

        return header;
    }

    static int64_t CalculateSize(int contentMetaCount) NN_NOEXCEPT
    {
        return sizeof(Header) + sizeof(ncm::ContentMetaKey) * contentMetaCount;
    }

    util::optional<fs::FileHandle> m_File;
    Header m_Header;
};

class FileApplyDownloadedDeltaTaskData
{
public:
    FileApplyDownloadedDeltaTaskData() NN_NOEXCEPT {}
    ~FileApplyDownloadedDeltaTaskData() NN_NOEXCEPT {}

    static Result Create(const char* path, int totalContentMeta) NN_NOEXCEPT;
    Result Initialize(const char* path, FileApplyDownloadedDeltaTaskMeta* meta) NN_NOEXCEPT;

    ncm::ApplyDeltaTaskBase::TaskState* GetTaskState() NN_NOEXCEPT { return &m_ApplyTaskState; }
    ncm::StorageContentMetaKey GetSourceKey() NN_NOEXCEPT
    {
        ncm::StorageContentMetaKey key = {m_Data.sourceKey, m_Meta->GetStorageId() };
        return key;
    }
    ncm::StorageContentMetaKey GetDestinationKey() NN_NOEXCEPT
    {
        switch(m_Meta->GetMetaVersion())
        {
        case FileApplyDownloadedDeltaTaskMeta::MetaVersion::Version0:
            {
                ncm::StorageContentMetaKey key = {m_Data.destinationKey, m_Meta->GetStorageId() };
                return key;
            }
        case FileApplyDownloadedDeltaTaskMeta::MetaVersion::Version1:
            {
                ncm::StorageContentMetaKey key = {m_Data.destinationKey, m_Data.destinationStorageId };
                return key;
            }
        default:
            {
                ncm::StorageContentMetaKey key = {};
                return key;
            }
        }
    }
    ApplyDownloadedDeltaProgressState GetState() NN_NOEXCEPT { return m_Data.applyDownloadedState; }
    int64_t GetRequiredSize() NN_NOEXCEPT { return m_Data.requiredSize; }
    bool IsTaskStateAvailable() NN_NOEXCEPT { return m_Data.taskStateAvailable; }
    bool IsFirstApply() NN_NOEXCEPT { return m_Data.currentIndex == 0; }

    Result SetState(ApplyDownloadedDeltaProgressState state) NN_NOEXCEPT;
    Result SetRequiredSize(int64_t size) NN_NOEXCEPT;
    Result SetTaskStateAvailable(bool flag) NN_NOEXCEPT;
    Result SetLastResult(Result result) NN_NOEXCEPT;
    ApplyDownloadedDeltaProgress GetProgress(const util::optional<ncm::ApplyDeltaProgress>& ncmProgress) NN_NOEXCEPT;
    Result MoveNext(bool* outComplete) NN_NOEXCEPT;
    Result Flush() NN_NOEXCEPT;
    void ResetThroughput() NN_NOEXCEPT;
    void BeginMeasurement() NN_NOEXCEPT { m_BeginMeasurementTick = os::GetSystemTick(); }
    void EndMeasurement() NN_NOEXCEPT;
    ApplyDeltaThroughput GetThroughput() NN_NOEXCEPT;

private:
    struct Data
    {
        ncm::ContentMetaKey                sourceKey;
        ncm::ContentMetaKey                destinationKey;
        ApplyDownloadedDeltaProgressState  applyDownloadedState;
        bool                               taskStateAvailable;
        ncm::StorageId                     destinationStorageId;
        Bit8                               reserved[5];
        int64_t                            applied;
        int64_t                            requiredSize;
        int                                currentIndex;
        int                                totalContentMeta;
        util::TypedStorage<Result, sizeof(Result), NN_ALIGNOF(Result)> lastResult;
    };

    FileApplyDownloadedDeltaTaskMeta*  m_Meta;
    Data                               m_Data;
    ncm::ApplyDeltaTaskBase::TaskState m_ApplyTaskState;
    char m_Path[64];

    // 速度計測用、永続化はしない
    int64_t   m_InitialProcessedSize; // 初期の進捗状況
    int64_t   m_TotalExecuteTime;     // 総実行時間 (msec)
    os::Tick  m_BeginMeasurementTick; // 計測開始 Tick
};


class ApplyDownloadedDeltaTask
{
    NN_DISALLOW_COPY(ApplyDownloadedDeltaTask);
    NN_DISALLOW_MOVE(ApplyDownloadedDeltaTask);

public:
    ApplyDownloadedDeltaTask() NN_NOEXCEPT : m_ApplyTaskProgressAvailable(false) {}
    ~ApplyDownloadedDeltaTask() NN_NOEXCEPT {}

    Result Initialize(const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT;
    Result InitializeBeforeRun() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
    void FinalizeAfterRun() NN_NOEXCEPT{}
    Result InitialPrepare() NN_NOEXCEPT;
    Result PrepareAndExecute() NN_NOEXCEPT;
    ApplyDownloadedDeltaProgress GetProgress() NN_NOEXCEPT;

    Result Commit(bool doCleanup = true) NN_NOEXCEPT;
    Result CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT;
    Result CalculateOccupiedSize(int64_t* outValue, ncm::StorageId storageId) NN_NOEXCEPT;
    void GetRequiredStorage(ncm::StorageId* outValue) NN_NOEXCEPT { *outValue = m_Meta.GetStorageId(); }
    Result Cleanup() NN_NOEXCEPT;
    Result Cancel() NN_NOEXCEPT;
    Result ResetCancel() NN_NOEXCEPT;
    Result ClearNotEnoughSpaceState() NN_NOEXCEPT;
    Result SetBuffer(void* buffer, size_t size) NN_NOEXCEPT;
    ncm::ApplicationId GetApplicationId() NN_NOEXCEPT;
    Result ListContentMetaKey(int* outCount, ncm::StorageContentMetaKey* outList, int count, int offset) NN_NOEXCEPT;
    ApplyDeltaThroughput GetThroughput() NN_NOEXCEPT { return m_Data.GetThroughput(); }
    void GetErrorContext(err::ErrorContext* outValue) NN_NOEXCEPT
    {
        *outValue = {};
    }

private:
    Result InitialPrepareImpl() NN_NOEXCEPT;
    Result PrepareAndExecuteImpl() NN_NOEXCEPT;
    Result CommitImpl(bool doCleanup) NN_NOEXCEPT;
    Result MoveNext(bool* outComplete) NN_NOEXCEPT;
    Result ThrowIfCancelRequested() NN_NOEXCEPT;

    bool                             m_IsTaskInitialized;
    bool                             m_CancelToken;
    bool                             m_ApplyTaskProgressAvailable;

    FileApplyDownloadedDeltaTaskMeta m_Meta;
    FileApplyDownloadedDeltaTaskData m_Data;
    ncm::ApplyPatchDeltaTask         m_ApplyTask;

};

}}}
