﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/os.h>
#include <nn/lmem/lmem_ExpHeap.h>

namespace nn { namespace nim {

/**
 * @brief   リソースコンフィギュレーション
 */
struct ShopServiceAccessConfiguration
{
    static const size_t ExpHeapManagedBlockSize = (sizeof(::nn::lmem::HeapCommonHead) + std::alignment_of<std::max_align_t>::value) & (~std::alignment_of<std::max_align_t>::value);
    static const size_t IServiceObjectSize = 192;    // > sizeof(sf::detail::ObjectImplFactoryWithStatefulAllocator<Base, sf::ExpHeapAllocator>::Object);

    /**
     * @brief   同時接続可能なクライアントプロセス上限数。
     */
    static const size_t AvailableServiceClientMax = 1;

    /**
     * @brief   セッション維持対象アクセッサ上限数。
     * @details クライアントプロセス単位の上限数です。@n
     *          @ref AvailableServiceClientMax 倍数の容量確保される想定です。
     * @see     RequiredSessionCount
     */
    static const size_t AvailableAccessorMax = 1;

    /**
     * @brief   セッション維持対象非同期アクセスハンドル上限数。
     * @details クライアントプロセス単位の非同期ハンドル上限数です。@n
     *          @ref AvailableServiceClientMax 倍数の容量確保される想定です。
     * @see     RequiredSessionCount
     */
    static const size_t AvailableAsyncAccessMax = 16;

    /**
     * @brief   セッション維持最大要求数
     * @note    IShopServiceAccessServerInterface と IShopServiceAccessServer 用セッションは１つとして考えます。
     *          IShopServiceAccessServerInterface から IShopServiceAccessServer を取得後、即破棄することで除外可能。
     */
    static const size_t RequiredSessionCount = AvailableServiceClientMax
        + (AvailableAccessorMax * AvailableServiceClientMax)
        + (AvailableAsyncAccessMax * AvailableServiceClientMax);

    /**
     * @brief   並列通信最大数。(サービスプロセス側がクライアント用に準備できる最大スレッド数と同意です。)
     * @note    クライアントごとのリクエストスケジューラが準備するスレッドの最大スレッド数を意味します。
     */
    static const size_t ParallelExecutionCount = 1;

    /**
     * @brief   キューバッファ要素上限数
     */
    static const size_t RequestQueueCapacity = AvailableAsyncAccessMax;

    /**
     * @brief   要求実行作業スレッドスタックサイズ( byte単位 )
     * @note     実計測( Horizon )消費:   15～16 KiB
     */
    static const size_t EnsureWorkerThreadStackSize = ::nn::os::MemoryPageSize * 5;

    /**
     * @brief   要求実行監視スレッドスタックサイズ( byte単位 )
     * @note     実計測( Horizon )消費:   2KiB
     */
    static const size_t EnsureSchedulerThreadStackSize = ::nn::os::MemoryPageSize * 1;
};

}}
