﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_Request.h>


namespace nn
{
namespace nifm
{

/**
 * @brief   ネットワーク接続を扱うためのクラスです。
 */
class NetworkConnection
{
    NN_DISALLOW_COPY(NetworkConnection);
    NN_DISALLOW_MOVE(NetworkConnection);

private:
    nn::nifm::Request m_Request;

public:
    /**
     * @brief   標準的な設定のインターネット接続を扱うためのオブジェクトを構築します。
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  デフォルトコンストラクタです。@n
     *
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  NetworkConnection クラスのオブジェクトを大域的に配置することはできません。@n
     *
     *  GetSystemEvent 関数で取得されるイベントのクリアモードは自動に設定されます。
     */
    NetworkConnection() NN_NOEXCEPT;

    /**
     * @brief   標準的な設定のインターネット接続を扱うためのオブジェクトを構築します。
     *
     * @param[in]  eventClearMode   GetSystemEvent 関数で取得されるシステムイベントのクリアモードを指定します
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  NetworkConnection クラスのオブジェクトを大域的に配置することはできません。
     */
    explicit NetworkConnection(nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief  デストラクタです。
     *
     * @details
     *  オブジェクトを破棄した場合、そのオブジェクトを通じて提出していた要求も取り下げられます。
     *  これにより、システムのネットワーク接続が切断される可能性があることに注意してください。
     *  ネットワーク接続を維持したい間は、要求を提出したオブジェクトを維持する必要があります。
     */
    ~NetworkConnection() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク接続をシステムに要求します。
     *
     * @details
     *  本関数を呼ぶと非同期処理が発生し、ネットワーク接続を利用できるかの検証がおこなわれます。
     *  検証中は IsRequestOnHold 関数が true を返します。
     *  アプリケーションは本 API のコール後、定期的に IsRequestOnHold の状態を確認し、
     *  IsRequestOnHold 関数が false を返したら
     *  IsAvailable 関数でネットワーク接続が利用できる状態になったかを確認してください。
     *  IsRequestOnHold 関数が true を返すあいだに利用要求を取り下げると、利用可能なネットワークがあるにもかかわらずネットワーク接続を利用できないことがあるため、この間はユーザー操作以外でのキャンセルはおこなわないことを推奨します。@n
     *
     *  ネットワーク接続の利用要求の検証は本 API のコール1回につき1回のみおこなわれます。
     *  検証完了後あらためてネットワーク接続の利用を要求するには、もう一度本 API を呼び直す必要があります。@n
     *
     *  この関数はブロックすることなくすぐに返ります。@n
     *
     *  この関数は SubmitRequestAndWait 関数とスレッドアンセーフです。
     */
    void SubmitRequest() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク接続をシステムに要求し、要求が受理もしくは却下されるまでブロックします。
     *
     * @details
     *  本関数を呼ぶとネットワーク接続を利用できるかの検証がおこなわれ、検証の結果が出るまでブロックします。
     *  本関数を抜けたら IsAvailable 関数でネットワーク接続が利用できる状態になったかを確認してください。@n
     *
     *  ネットワーク接続の利用要求の検証は本 API のコール1回につき1回のみおこなわれます。
     *  検証完了後あらためてネットワーク接続の利用を要求するには、もう一度本 API を呼び直す必要があります。@n
     *
     *  この関数を抜けたあと、 GetSystemEvent 関数から取得されるイベントはシグナル状態になっていることに注意してください。@n
     *
     *  システムの内部状態によって、検証処理はごく短時間で完了することがあります。@n
     *
     *  この関数は SubmitRequest 関数とスレッドアンセーフです。
     */
    void SubmitRequestAndWait() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク接続の要求を取り下げます。
     *
     * @details
     *  システムに提出していたネットワーク接続の要求を取り下げます。@n
     *
     *  この関数は要求が検証中か否かに関わらず呼ぶことができます。
     *  SubmitRequestAndWait 関数のブロックを解除したい場合、この関数を別スレッドから呼んでください。@n
     *
     *  この関数を抜けたあと、 GetSystemEvent 関数から取得されるイベントはシグナル状態になっていることに注意してください。@n
     *
     *  この関数はネットワーク接続に興味をなくしたことを宣言するだけであり、ネットワーク接続の切断を要求するものではありません。
     */
    void CancelRequest() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク接続の利用要求が検証中であるかどうかを取得します。
     *
     * @return  要求が検証中であれば true を、そうでなければ false を返します。
     *
     * @details
     *  システムの内部状態によって、検証処理はごく短時間で完了することがあります。
     */
    bool IsRequestOnHold() NN_NOEXCEPT;

    /**
     * @brief   ネットワーク接続が利用可能であるかを取得します。
     *
     * @return  ネットワーク接続が利用可能であれば true を、そうでなければ false を返します。
     *
     * @details
     *  この関数が true を返す状態であれば、ネットワーク接続を利用することができます。@n
     *
     *  無線 LAN アクセスポイントからの切断などの外部要因により、予期せず
     *  true から false へ遷移することがあることに注意してください。
     *  逆に、 SubmitRequest 関数もしくは SubmitRequestAndWait 関数による利用要求の提出を経ずして
     *  false から true へ遷移することはありません。
     */
    bool IsAvailable() NN_NOEXCEPT;

    /**
     * @brief   利用要求の状態変化を待ち受けるためのシステムイベントを取得します。@n
     *
     * @return  nn::os::SystemEvent オブジェクトへの参照を返します。
     *
     * @details
     *  利用要求の状態が変化したときにシグナルされるシステムイベントへの参照を取得します。
     *  この関数が返すイベントがシグナルした場合、 IsRequestOnHold 関数や IsAvailable 関数でネットワーク接続の状態を確認してください。@n
     *
     *  イベントのクリアモードは、本クラスのコンストラクタに指定する引数によって決定されます。
     */
    nn::os::SystemEvent& GetSystemEvent() NN_NOEXCEPT;


    /**
     * @brief 利用要求提出の結果に対するハンドリングを行います。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 利用要求は受理済みです。
     *   }
     *   @handleresult{
     *     ResultErrorHandlingCompleted, 利用要求は受理されていませんでしたが、エラーハンドリングが正常に行われました。エラーが解消された可能性があります。
     *   }
     *   - その他の Result の場合、利用要求は受理されておらず、エラーハンドリングにも失敗しています。
     * @endretresult
     *
     * @details
     *  エラーに応じて、ハンドリング中にユーザー操作が必要になる場合があり、長時間ブロックする可能性があります。@n
     *
     *  この関数が成功を返した場合、利用要求は受理されておりネットワークを利用できます。@n
     *  この関数が失敗を返した場合、利用要求は却下されており、
     *    - 返された失敗が ResultErrorHandlingCompleted の場合、利用要求を再提出するとすぐに受理される可能性があります。@n
     *      再提出した要求が却下された場合でも、本 API を再度呼び出すことが可能です。
     *    - 返された失敗が その他の Result の場合、利用要求の再提出に対する結果を予測できないため、直前のシーケンスに戻ることを推奨します。
     *
     *  本 API によって、エラービューアの表示などが適宜行われますので、呼び出し側がエラーコードなどを利用者に提示する必要はありません。
     *
     *  また、ユーザー操作などにより利用要求が明示的に取り下げられた場合は、この関数を呼んでエラーの解消を試みる必要はありません。
     *
     *  @platformbegin{Windows}
     *  Windows 版では、利用要求が却下された際の本 API 呼び出しによって ResultErrorHandlingCompleted が返ることはありません。@n
     *  利用要求が受理されている場合は、通常通り ResultSuccess を返します。
     *  @platformend
     */
    nn::Result HandleNetworkRequestResult() NN_NOEXCEPT;

    // for private use
    RequestHandle GetRequestHandle() NN_NOEXCEPT;

    // for internal use
    nn::Result GetResult() NN_NOEXCEPT;
    RequestState GetRequestState() NN_NOEXCEPT;
};

}
}
