﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesRequestPrivate.h>
#include <nn/nifm/nifm_TypesRequirement.h>

// このヘッダ内の API は、パッケージとして提供されていません。
// 提供する場合は、個別に切り出したヘッダを作成するように。

namespace nn
{
namespace nifm
{
    //! @name 利用要求編集 API
    //! @{

    /**
     * @brief 一度受理された利用要求がシステムに却下され得るか否かを指定します。
     *
     * @param[in] handle       利用要求ハンドル
     * @param[in] isRejectable 却下可能フラグ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  却下可能フラグを false に設定した利用要求が一度受理されると、それがシステムに却下されることはありません。
     *  この利用要求の取り下げが必要になった場合、 nn::nifm::Request::GetRequestState で RequestState_Blocking という状態を取得できます。
     *  この状態では、必要な終了処理を行った後 nn::nifm::Request::Cancel によって利用要求を取り下げる必要があります。
     */
    nn::Result SetRequestRejectable(RequestHandle handle, bool isRejectable) NN_NOEXCEPT;

    /**
     * @brief 利用要求が提出後即時評価を求めるか否かを指定します。
     *
     * @param[in] handle        利用要求ハンドル
     * @param[in] isInstant     即時評価フラグ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  即時評価フラグを true に設定した利用要求は原則として即時評価されますが、別の利用要求の評価が進行中の場合は遅延することがあります。
     *  即時評価フラグを false に設定した利用要求は、別の理由で提出済みの利用要求の評価がおこなわれる際にまとめて評価されます。
     *
     *  この関数はテスト用です。
     */
    nn::Result SetRequestInstant(RequestHandle handle, bool isInstant) NN_NOEXCEPT;

    /**
     * @brief 利用要求が内部的に複製され、要求取り下げ後も接続を維持し続けるか否かを指定します。
     *
     * @param[in] handle           利用要求ハンドル
     * @param[in] isSustainable    接続維持可能フラグ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  接続維持可能フラグを true に設定した場合、要求提出時に内部的に要求が複製されます。
     *  接続維持可能な利用要求が提出されると、元の要求が取り下げられた場合でも複製された要求が残り続け、接続を維持します。
     *  ただし再度、接続維持可能フラグが true な要求が提出されると、以前の複製された要求は却下されます。
     *  ローカル通信または独占の利用要求に対して本フラグを設定した場合、提出時に失敗します。
     */
    nn::Result SetRequestSustainable(RequestHandle handle, bool isSustainable) NN_NOEXCEPT;

    /**
    * @brief 利用要求の優先度を直値で設定します。
    *
    * @param[in] handle    利用要求ハンドル
    * @param[in] rawPriority  優先度
    *
    * @retresult
    *   @handleresult{
    *     nn::ResultSuccess, 処理に成功しました。
    *   }
    * @endretresult
    *
    * @pre
    *   - ライブラリが初期化済み
    *   - handle が有効な値
    *   - handle が表す利用要求の提出前または取り下げ後
    *
    * @details
    *  優先度は、数値が小さいほど高い優先度を表します。
    */
    nn::Result SetRequestRawPriority(RequestHandle handle, uint8_t rawPriority) NN_NOEXCEPT;

    /**
     * @brief 利用要求が、帯域共有不可能な他の要求を排除してより大きい帯域を使用するか否か、を設定します。
     *
     * @param[in] handle    利用要求ハンドル
     * @param[in] isGreedy  帯域要求フラグ
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  帯域要求フラグが true の利用要求と、nn::nifm::SetRequestSharable で設定される帯域共有可能フラグが false の利用要求は、共存できません。
     *  つまり、大きな帯域を欲する要求と、大きな帯域を分け合うことができない要求とは共存できません。
     *  また、帯域共有可能フラグが false の要求同士でも、どちらも大きな帯域を要求しない、つまり帯域要求フラグが false の場合は共存可能です。
     *  共存可能な組み合わせは、次の三パターンに分類されます。
     *
     *  - !greedy && sharable はすべての要求と共存可能
     *  -  greedy && sharable 同士は共存可能
     *  - !greedy 同士は共存可能
     *
     *  帯域要求フラグと共有要求フラグのどのような組み合わせで共存が可能なのかを示したものが下記の表です。
     *
     *  |                      |  greedy && sharable  |  greedy && !sharable | !greedy && sharable  | !greedy && !sharable |
     *  |:--------------------:|:--------------------:|:--------------------:|:--------------------:|:--------------------:|
     *  |  greedy &&  sharable | 共存可能             |                      | 共存可能             |                      |
     *  |  greedy && !sharable |                      |                      | 共存可能             |                      |
     *  | !greedy &&  sharable | 共存可能             | 共存可能             | 共存可能             | 共存可能             |
     *  | !greedy && !sharable |                      |                      | 共存可能             | 共存可能             |
     *
     */
    nn::Result SetRequestGreedy(RequestHandle handle, bool isGreedy) NN_NOEXCEPT;

    /**
    * @brief 利用要求が、大きい帯域を要求する利用要求と帯域を共有可能であるか否か、を設定します。
    *
    * @param[in] handle      利用要求ハンドル
    * @param[in] isSharable  帯域共有可能フラグ
    *
    * @retresult
    *   @handleresult{
    *     nn::ResultSuccess, 処理に成功しました。
    *   }
    * @endretresult
    *
    * @pre
    *   - ライブラリが初期化済み
    *   - handle が有効な値
    *   - handle が表す利用要求の提出前または取り下げ後
    *
    * @details
    *  帯域共有可能フラグが false の利用要求と、 nn::nifm::SetRequestGreedy で指定される帯域要求フラグが true の利用要求は、共存できません。
    *  詳細は nn::SetRequestGreedy の説明を参照してください。
    */
    nn::Result SetRequestSharable(RequestHandle handle, bool isSharable) NN_NOEXCEPT;

    /**
     * @brief 利用要求がスリープ遷移時にも取り下げずに維持するよう求めるか、を設定します。
     *
     * @param[in] handle        利用要求ハンドル
     * @param[in] isKeptInSleep スリープ中維持フラグ
     *
     * @return 処理の結果を表す nn::Result 値を返します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     *  この設定を有効にしていても、 nn::nifm::SetRequestSocketDescriptor() で維持するべきセッションの登録をおこなっていなかった場合、スリープ遷移時にネットワーク接続が切断され、利用要求が取り下げられます。@n
     *  また、システム上の制約により接続が維持できない状況下では、この関数での設定にかかわらず、スリープ遷移時にネットワーク接続が切断され、利用要求が取り下げられることがあります。@n
     */
    nn::Result SetRequestKeptInSleep(RequestHandle handle, bool isKeptInSleep) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を登録します。
     *
     * @param[in] handle            利用要求ハンドル
     * @param[in] socketDescriptor  維持するべきセッションのソケットディスクリプタ
     *
     * @return 処理の結果を表す nn::Result 値を返します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *
     * @details
     *  スリープ中にネットワーク接続を維持するには、別途 nn::nifm::SetRequestRequirementPreset() もしくは
     *  nn::nifm::SetRequestKeptInSleep() で事前に設定をおこなう必要があります。@n
     *  @n
     *  この関数は利用要求が受理された状態で呼ばないと失敗を返します。@n
     *  @n
     *  この関数で登録されたセッションの情報は、利用要求の却下時に破棄されます。
     */
    nn::Result RegisterRequestSocketDescriptor(RequestHandle handle, int socketDescriptor) NN_NOEXCEPT;

    /**
     * @brief スリープ中に維持するべきセッションの情報を破棄します。
     *
     * @param[in] handle            利用要求ハンドル
     *
     * @return 処理の結果を表す nn::Result 値を返します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *
     * @details
     *  この関数は利用要求が受理された状態で呼ばないと失敗を返します。
     */
    nn::Result UnregisterRequestSocketDescriptor(RequestHandle handle) NN_NOEXCEPT;

    /**
     * @brief 利用要求に紐づいた追加応答情報を取得します。
     *
     * @param[out] pOutAdditionalInfo   追加応答情報
     * @param[out] pOutRevision         利用要求提出番号
     * @param[in] handle                利用要求ハンドル
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 処理に成功しました。
     *   }
     * @endretresult
     *
     * @pre
     *   - ライブラリが初期化済み
     *   - handle が有効な値
     *   - handle が表す利用要求の提出前または取り下げ後
     *
     * @details
     */
    nn::Result GetRequestAdditionalInfo(AdditionalInfo* pOutAdditionalInfo, uint32_t* pOutRevision, RequestHandle handle) NN_NOEXCEPT;

    //! @}
}
}

