﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/core/profile/nifm_WirelessUserSetting.h>
#include <nn/nifm/detail/core/profile/nifm_WirelessServiceSetting.h>
#include <nn/nifm/detail/core/profile/nifm_UserNetworkProfile.h>
#include <nn/nifm/detail/core/profile/nifm_SsidListNetworkProfile.h>
#include <nn/nifm/detail/core/profile/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/detail/core/profile/nifm_DummyNetworkProfile.h>
#include <nn/nifm/detail/util/nifm_ArrayedFactory.h>
#include <nn/nifm/detail/util/nifm_EventHandler.h>

#include <nn/settings/system/settings_Network.h>

#include <nn/os/os_SdkMutex.h>


namespace nn
{
namespace nifm
{

struct IpSettingData;

namespace detail
{

class NetworkProfileManager
{
    NN_DISALLOW_COPY(NetworkProfileManager);
    NN_DISALLOW_MOVE(NetworkProfileManager);

public:
    static const uint32_t InvalidId = 0;
    static const DummyNetworkProfile s_DummyNetworkProfile;

private:
    mutable nn::os::SdkMutex m_Mutex;

    nn::os::Event m_NetworkProfileEvent;
    SingleEventHandler m_NetworkProfileEventHandler;

    class NetworkProfileEventCallback : public CallbackObject
    {
    public:
        explicit NetworkProfileEventCallback() {}

    private:
        void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE {}
    } m_NetworkProfileEventCallback;

    static const size_t UserNetworkProfileSize = UserNetworkProfileCountMax;
    using UserNetworkProfileList = ArrayedFactory<UserNetworkProfile, UserNetworkProfileSize>;
    static const size_t SsidListNetworkProfileSize = SsidListNetworkProfileCountMax;
    using SsidListProfileList = ArrayedFactory<SsidListNetworkProfile, SsidListNetworkProfileSize>;
    static const size_t TemporaryNetworkProfileSize = 1;
    using TemporaryNetworkProfileList = ArrayedFactory<TemporaryNetworkProfile, TemporaryNetworkProfileSize>;

private:
    // ユーザー設定
    UserNetworkProfileList m_UserNetworkProfileList;
    // SSID リスト
    SsidListProfileList m_SsidListNetworkProfileList;
    // テンポラリ接続用
    TemporaryNetworkProfileList m_TemporaryNetworkProfileList;

    // SSID リストバージョン
    SsidListVersion m_SsidListVersion;

    // [TORIAEZU]
    UserNetworkProfileList m_CopiedUserNetworkProfileList;
    SsidListProfileList m_CopiedSsidListNetworkProfileList;
    TemporaryNetworkProfileList m_CopiedTemporaryNetworkProfileList;

    nn::settings::system::NetworkSettings m_NetworkSettings[UserNetworkProfileCountMax];

private:
    nn::Result ConvertToNetworkProfileBasicInfoFromNetworkProfile(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, const NetworkProfileBase& networkProfile) const NN_NOEXCEPT;

    // 渡された ID が設定 ID として有効かを確認し、無効ならば有効な ID に変換して返す
    bool ConfirmValidProfileId(nn::util::Uuid* id) NN_NOEXCEPT;

    nn::Result RemoveNetworkProfileImpl(NetworkProfileBase::ProfileType type, const nn::util::Uuid& id) NN_NOEXCEPT;

public:
    NetworkProfileManager() NN_NOEXCEPT;
    virtual ~NetworkProfileManager() NN_NOEXCEPT;

    nn::Result UpdateUserNetworkProfile(const NetworkProfileData& networkProfile) NN_NOEXCEPT;

    nn::Result AddUserNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT;
    nn::Result AddSsidListNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT;
    nn::Result AddTemporaryNetworkProfile(nn::util::Uuid *pOutId, const NetworkProfileData& networkProfile) NN_NOEXCEPT;

    nn::Result RemoveNetworkProfile(NetworkProfileBase::ProfileType type, const nn::util::Uuid& id ) NN_NOEXCEPT;

    void RemoveAllNetworkProfile(NetworkProfileBase::ProfileType type) NN_NOEXCEPT;

    int GetNetworkProfileCount(NetworkProfileBase::ProfileType type) const NN_NOEXCEPT;

    nn::Result GetNetworkProfileBasicInfo(NetworkProfileBasicInfo* pOutNetworkProfileBasicInfo, NetworkProfileBase::ProfileType type, int index) const NN_NOEXCEPT;
    nn::Result GetNetworkProfileData(NetworkProfileData* pOutNetworkProfileData, const nn::util::Uuid& id) const NN_NOEXCEPT;

    // 有線の利用可能な接続設定があるか確認し、あれば true を、なければ false を返します
    // 自動接続で利用しない指定がされている接続設定は対象に含まれません
    bool IsAnyEthernetNetworkProfileAvailable() const NN_NOEXCEPT;

    EventHandler& GetNetworkProfileEventHandler() NN_NOEXCEPT
    {
        return m_NetworkProfileEventHandler;
    }

    // 接続成功時に呼び出して，接続設定の順序を更新する
    bool UpdateUserNetworkProfileOrder( const nn::util::Uuid& id ) NN_NOEXCEPT;

    // 接続設定を Settings から読み込む
    nn::Result ImportUserNetworkProfile() NN_NOEXCEPT;
    // 接続設定を Settings に書き込む
    nn::Result ExportUserNetworkProfile() NN_NOEXCEPT;

    // 接続設定を SSID リストから読み込む
    nn::Result ImportSsidListNetworkProfile() NN_NOEXCEPT;

    // SSID リストのバージョンを取得する
    nn::Result GetSsidListVersion(SsidListVersion* pOutSsidListVersion) const NN_NOEXCEPT;

    nn::Result PerpetuateTemporaryNetworkProfile(nn::util::Uuid* pOutId, const nn::util::Uuid& id) NN_NOEXCEPT;

    // TODO: 接続設定関連の排他処理を行うまでの一時的処置
    nn::Result CopyAllNetworkProfiles() NN_NOEXCEPT;
    const NetworkProfileBase* GetCopiedNetworkProfile( NetworkProfileBase::ProfileType type, int index ) const NN_NOEXCEPT;
    int GetCopiedNetworkProfileCount( NetworkProfileBase::ProfileType type ) const NN_NOEXCEPT;

    // for Debug
    void Dump() const NN_NOEXCEPT;
    void GetDefaultIpSetting(IpSettingData* pOutIpSetting) NN_NOEXCEPT;
    void SetDefaultIpSetting(const IpSettingData& ipSetting) NN_NOEXCEPT;
};

}
}
}
