﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/nifm/detail/util/nifm_Heap.h>

#include <nn/util/util_IntrusiveList.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class AccessPointBase;
class WirelessInterface;

using AccessPointIntrusiveList = nn::util::IntrusiveList<AccessPointBase, nn::util::IntrusiveListBaseNodeTraits<AccessPointBase>>;

class AccessPointListBase : public AccessPointIntrusiveList
{
    NN_DISALLOW_COPY(AccessPointListBase);
    NN_DISALLOW_MOVE(AccessPointListBase);

protected:
    static const nn::TimeSpan DefaultLifetime;

private:
    HeapBase& m_Heap;
    AccessPointIntrusiveList m_TempList;
    nn::TimeSpan m_Lifetime;

public:
    AccessPointListBase(HeapBase& heap, nn::TimeSpan lifetime) NN_NOEXCEPT;
    virtual ~AccessPointListBase() NN_NOEXCEPT;

    void Clear() NN_NOEXCEPT;

    template<typename T, typename... Ts>
    T* CreateAndAdd(Ts... args) NN_NOEXCEPT
    {
        auto p = m_Heap.Create<T>(args...);

        if (p != nullptr)
        {
            AddToListImpl(*p, *this);
        }

        return p;
    }

    // アクセスポイントリストの保持するヒープからメモリを確保してオブジェクトを生成するがリストには追加しない
    // これで生成したオブジェクトは、 Commit() で確定するまで、 Create() 時の重複判定や DeleteOldest() の対象に入らない
    template<typename T, typename... Ts>
    T* CreateTempAndAdd(Ts... args) NN_NOEXCEPT
    {
        auto p = m_Heap.Create<T>(args...);

        if (p != nullptr)
        {
            AddToListImpl(*p, m_TempList);
        }

        return p;
    }

    template<typename T, typename... Ts>
    T* CreateAndUpdate(Ts... args) NN_NOEXCEPT
    {
        auto p = m_Heap.Create<T>(args...);

        if (p != nullptr)
        {
            UpdateListImpl(*p, *this);
        }

        return p;
    }

    // アクセスポイントリストの保持するヒープからメモリを確保してオブジェクトを生成するがリストには追加しない
    // これで生成したオブジェクトは、 Commit() で確定するまで、 Create() 時の重複判定や DeleteOldest() の対象に入らない
    template<typename T, typename... Ts>
    T* CreateTempAndUpdate(Ts... args) NN_NOEXCEPT
    {
        auto p = m_Heap.Create<T>(args...);

        if (p != nullptr)
        {
            UpdateListImpl(*p, m_TempList);
        }

        return p;
    }

    // CreateTemp() で追加したアクセスポイントをリストに追加する
    void Commit() NN_NOEXCEPT;

    // アクセスポイントリストから、登録順が最も古くて最も優先度の低いものを削除してヒープを空ける。
    // 一度のスキャンで複数のアクセスポイントが登録された場合、これらは同程度に古いものとみなされタイムスタンプは関係しない。
    bool DeleteOldest() NN_NOEXCEPT;

    // 保持するアクセスポイントを pAccessPointList で渡したリストにコピーします
    // pAccessPointList が保持していたアクセスポイントはそのまま残ります
    // pAccessPointList がいっぱいになった場合はそこでコピーを終了します
    void ExportTo(AccessPointListBase* pAccessPointList) const NN_NOEXCEPT;

private:
    void AddToListImpl(AccessPointBase& accessPoint, AccessPointIntrusiveList& list) NN_NOEXCEPT;
    void UpdateListImpl(AccessPointBase& accessPoint, AccessPointIntrusiveList& list) NN_NOEXCEPT;

    void push_back(AccessPointBase& value) NN_NOEXCEPT;
    void push_front(AccessPointBase& value) NN_NOEXCEPT;
    void pop_back() NN_NOEXCEPT;
    void pop_front() NN_NOEXCEPT;
    void clear() NN_NOEXCEPT;
};

template<typename HeapType>
class AccessPointList : public AccessPointListBase
{
    NN_DISALLOW_COPY(AccessPointList);
    NN_DISALLOW_MOVE(AccessPointList);

private:
    HeapType m_Heap;

public:
    explicit AccessPointList(nn::TimeSpan lifetime = DefaultLifetime) NN_NOEXCEPT
        : AccessPointListBase(m_Heap, lifetime)
    {
    }

    virtual ~AccessPointList()
    {
    }
};

}
}
}
