﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      NFP(amiibo) の機能検証に必要な非公開 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nfp/nfp_Result.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/nfp/nfp_Types.h>
#include <nn/nfp/nfp_DebugTypes.h>

namespace nn {
namespace nfp {

/**
 * @name    Read/Write 機能検証
 * @{
 */

/**
 * @brief   NFP ライブラリをデバッグモードで初期化します。
 *
 * @pre
 *  - GetState() == @ref State_None
 *
 * @post
 *  - NFP ライブラリがデバッグモードで初期化済みの状態になります。
 *  - GetState() == @ref State_Init
 *
 * @details
 * nfp_DebugApi.h で定義されたデバッグ用の API を使用するためには、
 * この API で NFP ライブラリを初期化する必要があります。@n
 * デバッグモードでは、一般公開 API、および、システムモード API も使用できます。
 */
void InitializeDebug() NN_NOEXCEPT;

/**
 * @brief   NFP ライブラリを終了します。
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  - GetState() == @ref State_None
 *
 * @details
 *  この関数は NFP ライブラリの使用を終える際に呼び出してください。
 *  InitializeDebug() で NFP ライブラリを初期化した場合には、この API でライブラリを終了します。
 */
void FinalizeDebug() NN_NOEXCEPT;

/**
 * @brief         NFP のタグデータを、マウント時に用意した内部バッファから取得します。
 *
 * @param[out]    pOutNfpData              取得したタグデータの出力先です。
 * @param[in]     deviceHandle             NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               取得に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           取得に失敗しました。取得対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 * タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 * この API では、オーナー登録情報やアプリケーション専用領域が論理的に存在しない場合でも、
 * 該当する領域のデータをそのまま取得します。
 * それに対して @ref GetCommonInfo(), @ref GetRegisterInfo(), @ref GetAdminInfo() では、
 * 存在しない情報を 0 で埋めて返します。
 * そのため、この API で取得した値が @ref GetCommonInfo(), @ref GetRegisterInfo(), @ref GetAdminInfo() とは異なる場合があります。
 */
nn::Result GetAll(NfpData* pOutNfpData, const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief         NFP のタグデータを、マウント時に用意した内部バッファに設定します。
 *
 * @param[in]     deviceHandle             NFC デバイスのハンドル
 * @param[in]     nfpData                  設定対象となるタグデータです。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               設定に成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           設定に失敗しました。設定対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  この API で内部バッファに設定した内容を実際にタグに書き込むためには、
 *  @ref FlushDebug() を呼び出してください。 @ref Flush() の方を使用すると、
 *  共用領域の一部がシステムによって自動的に更新されてしまうため、
 *  意図したデータをタグに書き込むことができなくなります。
 *
 * @details
 *  タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  以下フィールドについてはこの API を使用して書き換えることができません。
 *  - CommonInfo
 *   - characterId
 *   - seriesId
 *   - numberingId
 *   - nfpType
 *   - applicationAreaSize
 *  - AdminInfo
 *   - formatVersion
 *   - platform
 */
nn::Result SetAll(const DeviceHandle& deviceHandle, const NfpData& nfpData) NN_NOEXCEPT;

/**
 * @brief         マウント時に用意した内部バッファの内容をタグに書き込みます。
 *
 * @param[in]     deviceHandle       NFC デバイスのハンドル
 *
 * @retresult
 *     @handleresult{ResultSuccess,               書き込みに成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             書き込みに失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           書き込みに失敗しました。書き込み対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details
 * タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 * @ref Flush() とほぼ同じですが、システムが NFP タグの共用領域を自動的に変更しない点で異なります。
 * @ref SetAll() と併用することを想定しています。
 */
nn::Result FlushDebug(const DeviceHandle& deviceHandle) NN_NOEXCEPT;

/**
 * @brief         タグデータを破壊します。
 *
 * @param[in]     deviceHandle            NFC デバイスのハンドル
 * @param[in]     breakType               タグの壊し方を指定します。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               書き込みに成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             書き込みに失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           書き込みに失敗しました。書き込み対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 * Format() あるいは Restore() が必要な状態になります。
 *
 * @details
 * タグデータを指定した壊し方で破壊します。@n
 * タグをマウントしておく必要があります。その際、マウント対象に @ref MountTarget_Ram を含んでいる必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)
 */
nn::Result BreakTag(const DeviceHandle& deviceHandle, BreakType breakType) NN_NOEXCEPT;

/**
 * @brief   バックアップデータを読み込みます。
 *
 * @param[out] pOutBuffer         データの読み込み先バッファです。
 * @param[out] pOutSize           実際に読み込めたデータのサイズです。
 * @param[in]  bufferSize         読み込むデータのサイズです。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               読み込みに成功しました。}
 *     @handleresult{ResultBackupError,           バックアップデータへのアクセスに失敗しました。再試行によって成功する可能性があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pOutBuffer はアクセス可能なアドレスを指している必要があります。
 *  - @a pOutSize はアクセス可能なアドレスを指している必要があります。
 *  - 0 < @a bufferSize
 *
 * @details
 *  読み込めるデータの最大サイズは @ref BackupData のサイズです。
 *  このサイズ よりも小さい @a bufferSize を指定した場合、バックアップデータの先頭から一部だけが取得されます。
 */
nn::Result ReadBackupData(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT;

/**
 * @brief   バックアップデータを書き込みます。
 *
 * @param[in]  pData              書き込むデータです。
 * @param[in]  dataSize           設定するデータのサイズです。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               書き込みに成功しました。}
 *     @handleresult{ResultBackupError,           バックアップデータへのアクセスに失敗しました。再試行によって成功する可能性があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pData はアクセス可能なアドレスを指している必要があります。
 *  - 0 < @a dataSize && @a dataSize <= @ref BackupData のサイズ
 *
 * @details
 *  書き込むデータのサイズが @ref BackupData のサイズより小さい場合、残りの領域は変更されません。
 *
 */
nn::Result WriteBackupData(const void* pData, size_t dataSize) NN_NOEXCEPT;

/**
 * @brief   NTF をタグに書き込みます。
 *
 * @param[in]  deviceHandle       NFC デバイスのハンドル
 * @param[in]  pData              書き込む NTF データです。
 * @param[in]  dataSize           設定するデータのサイズです。
 * @param[in]  ntfWriteType       書き込み方を指定します。
 *
 * @retresult
 *     @handleresult{ResultSuccess,               書き込みに成功しました。}
 *     @handleresult{ResultNfcDeviceNotFound,     正しいステートの NFC デバイスが見つかりません。@ref ListDevices() による NFC デバイスのハンドル取得からやり直す必要があります。}
 *     @handleresult{ResultNfcDisabled,           NFC 機能が無効になっています。NFC 機能が有効になった後、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRetry,             書き込みに失敗しました。再試行によって成功する可能性があります。再試行にて 3 回失敗した場合は、@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNeedRestart,           書き込みに失敗しました。書き込み対象のタグを喪失した可能性があります。@ref StartDetection() によるタグの検知からやり直す必要があります。}
 *     @handleresult{ResultNotSupported,          未ロックの NTAG215 タグではありません。使用しているタグを確認する必要があります。}
 * @endretresult
 *
 * @pre
 *  - ライブラリが @ref InitializeDebug() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *  - @a pData はアクセス可能なアドレスを指している必要があります。
 *  - @a dataSize は NTF データのサイズ(672バイト)である必要があります。
 *
 * @details
 *  タグが検知済みでマウントも喪失もされていない必要があります。(状態が正しくない場合、@ref ResultNfcDeviceNotFound が返ります。)@n
 *  書き込みが可能なタグは未ロックの NTAG215 のみです。
 */
nn::Result WriteNtf(const DeviceHandle& deviceHandle, const void* pData, size_t dataSize, NtfWriteType ntfWriteType) NN_NOEXCEPT;

/**
 * @}
 */

}  // nfp
}  // nn
