﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/nfc/nfc_Types.h>
#include <nn/nfc/nfc_Result.h>
#include <nn/nfc/nfc_PrivateResult.h>
#include <nn/nfc/nfc_NfpResult.internal.h>
#include <nn/nfc/server/core/nfc_CoreService.h>
#include <nn/nfp/server/nfp_Device.h>
#include <nn/nfc/mifare/server/nfc_MifareDevice.h>
#include <nn/nfc/pt/server/nfc_PtDevice.h>

namespace nn { namespace nfc { namespace server {

class Device
{
    friend nn::nfp::server::Device;
    friend nn::nfc::mifare::server::Device;
    friend nn::nfc::pt::server::Device;

public:
    enum State
    {
        State_Init,          //!< タグを探していない状態です。
        State_Search,        //!< タグを探している状態です。
        State_Active,        //!< @ref State_Search でタグを検知するとこの状態になります。
        State_Deactive,      //!< タグを検知した後、タグが離れるとこの状態になります。
        State_Mount,         //!< @ref State_Active で近くにあるタグをマウントするとこの状態になります。
        State_Keep,          //!< @ref State_Active で近くにあるタグとのセッションを維持するとこの状態になります。
        State_Unexpected     //!< 想定していない状態です。
    };

public:
    Device(uint64_t uniqueId, const nn::nfc::DeviceHandle& deviceHandle, nn::os::MultiWaitType* eventMultiWait) NN_NOEXCEPT;
    ~Device() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Reset(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    Device::State GetState() NN_NOEXCEPT;
    void SetState(Device::State state) NN_NOEXCEPT;
    void SetSelectedTagId(const nn::nfc::TagId& tagId) NN_NOEXCEPT;
    nn::nfc::TagId GetSelectedTagId() NN_NOEXCEPT;
    void SignalActivateEvent() NN_NOEXCEPT;
    void SignalDeactivateEvent() NN_NOEXCEPT;
    void ClearDetectEvent(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    uint64_t GetUniqueId() NN_NOEXCEPT;
    nn::nfc::DeviceHandle GetHandle() NN_NOEXCEPT;
    nn::Result StartDetection(nn::nfc::server::core::Service* service, nn::nfc::NfcProtocol protocolFilter) NN_NOEXCEPT;
    nn::Result StopDetection(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result GetTagInfo(nn::nfc::TagInfo* pOutTagInfo, nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId, nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result Mount(nn::nfc::server::core::Service* service, nn::nfp::ModelType modelType, nn::nfp::MountTarget mountTarget) NN_NOEXCEPT;
    nn::Result Unmount() NN_NOEXCEPT;
    nn::Result OpenApplicationArea(nn::Bit32 accessId) NN_NOEXCEPT;
    nn::Result GetApplicationArea(void* pOutBuffer, size_t* pOutSize, size_t bufferSize) NN_NOEXCEPT;
    nn::Result SetApplicationArea(const void* pData, size_t dataSize) NN_NOEXCEPT;
    nn::Result RecreateApplicationArea(const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT;
    nn::Result Flush(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result Restore(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result CreateApplicationArea(nn::nfc::server::core::Service* service, const nn::nfp::ApplicationAreaCreateInfo& createInfo) NN_NOEXCEPT;
    nn::Result GetRegisterInfo(nn::nfp::RegisterInfo* pOutRegisterInfo) NN_NOEXCEPT;
    nn::Result GetCommonInfo(nn::nfp::CommonInfo* pOutCommonInfo) NN_NOEXCEPT;
    nn::Result GetModelInfo(nn::nfp::ModelInfo* pOutModelInfo) NN_NOEXCEPT;
    nn::Result Format(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result GetAdminInfo(nn::nfp::AdminInfo* pOutAdminInfo) NN_NOEXCEPT;
    nn::Result GetRegisterInfo(nn::nfp::RegisterInfoPrivate* pOutRegisterInfo) NN_NOEXCEPT;
    nn::Result SetRegisterInfo(const nn::nfp::RegisterInfoPrivate& regInfo) NN_NOEXCEPT;
    nn::Result DeleteRegisterInfo(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result DeleteApplicationArea(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result ExistsApplicationArea(bool* outValue) NN_NOEXCEPT;
    nn::Result GetAll(nn::nfp::NfpData* pOutNfpData) NN_NOEXCEPT;
    nn::Result SetAll(const nn::nfp::NfpData& nfpData) NN_NOEXCEPT;
    nn::Result FlushDebug(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result BreakTag(nn::nfc::server::core::Service* service, nn::nfp::BreakType breakType) NN_NOEXCEPT;
    nn::Result WriteNtf(nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, nn::nfp::NtfWriteType ntfWriteType) NN_NOEXCEPT;
    size_t GetApplicationAreaSize() NN_NOEXCEPT;
    nn::Result ReadMifare(nn::nfc::MifareReadBlockData* pOutBlockData, nn::nfc::server::core::Service* service, const nn::nfc::MifareReadBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT;
    nn::Result WriteMifare(nn::nfc::server::core::Service* service, const nn::nfc::MifareWriteBlockParameter* pBlockParameter, size_t blockCount) NN_NOEXCEPT;
    nn::Result SendCommandByPassThrough(void* pOutBuffer, size_t* pOutSize, nn::nfc::server::core::Service* service, const void* pData, size_t dataSize, size_t bufferSize, nn::TimeSpan timeout) NN_NOEXCEPT;
    nn::Result KeepPassThroughSession(nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    nn::Result ReleasePassThroughSession() NN_NOEXCEPT;

    nn::Result GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutXcdDeviceHandle, nn::nfc::server::core::Service* service) NN_NOEXCEPT;
    void Hide() NN_NOEXCEPT;
    bool IsHidden() NN_NOEXCEPT;
    void LoopEventThreadFunction(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

private:
    Device() NN_NOEXCEPT;
    nn::Result CheckActive() NN_NOEXCEPT;
    void DispatchEventInEventThread(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
    nn::Result AfterAccessTag(nn::nfc::server::core::Service* service, nn::nfc::server::core::ScopedSession* session, nn::Result result) NN_NOEXCEPT;

private:

    Device::State m_State;
    uint64_t m_UniqueId;
    nn::nfc::DeviceHandle m_Handle;

    //イベント多重待ち
    nn::os::MultiWaitType* m_pEventMultiWait;

    //イベントスレッド終了イベント
    nn::os::SystemEventType m_EventTerminateEvent;
    nn::os::MultiWaitHolderType m_EventTerminateHolder;

    //タグ検出・喪失イベント
    nn::os::SystemEventType m_DetectEvent;
    nn::os::MultiWaitHolderType m_DetectEventHolder;

    //タグ検出・喪失以外のイベント
    nn::os::SystemEventType m_Event;
    nn::os::MultiWaitHolderType m_EventHolder;

    //リセットイベント
    nn::os::SystemEventType m_ResetEvent;
    nn::os::MultiWaitHolderType m_ResetEventHolder;

    //
    nn::os::SystemEventType m_AccessFinishEvent;
    nn::os::SystemEventType m_AccessResetEvent;

    nn::nfc::TagId m_SelectedTagId;
    mutable nn::os::MutexType m_Mutex;
    bool m_IsRunningThread;
    int m_InitializedCount;
    nn::os::Tick m_HideStartTick;
    nn::nfc::server::core::Service* m_ActiveService;

    nn::nfp::server::Device m_Nfp;
    nn::nfc::mifare::server::Device m_NfcMifare;
    nn::nfc::pt::server::Device m_NfcPt;
};

}}}  // namespace nn::nfc::server
