﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/os/os_Types.h>
#include <nn/nfc/nfc_Types.h>
#include <nn/xcd/xcd_NfcTypes.h>

namespace nn { namespace nfc { namespace server { namespace core {

//-- xcd_NfcTypes.h

/**
 * @brief       NTAG シリーズのタグデータ情報です。
 */
struct NtagData
{
    nn::nfc::TagInfo tagInfo;                            //!< NFC タグの基本情報

    uint8_t         type2TagVersion;                     //!< Type2 タグのバージョン
    NN_PADDING3;
    char            signature[nn::xcd::Type2TagSignatureSize];    //!< Type2 タグの認証情報
    int32_t         blockCount;                          //!< 読み込んだブロック数
    struct
    {
        nn::xcd::NtagReadAddress address;                         //!< 読み込んだデータのアドレス情報
        char            data[nn::xcd::NtagReadBlockSizeMax];      //!< データ本体
    } readDataBlocks[nn::xcd::NtagReadBlockCountMax];             //!< 読み込んだデータ
};

/**
 * @brief       パススルーモードでの受信データです。
 */
struct PassThruData
{
    nn::nfc::TagInfo tagInfo;                          //!< NFC タグの基本情報
    char       responseData[nn::xcd::NfcPassThruDataSizeMax];      //!< 受信データ
    uint32_t   responseSize;                           //!< 受信サイズ
};

/**
 * @brief       NFC で受信する可能性あるデータを含んだ構造体です。
 *
 * @details     共用体内部には、 @a reason の値に応じた付加情報が格納されます。
 *              特に記述がない reason に対しては、付加情報はありません (共用体内部は all 0 になります)。
 */
struct Info
{
    nn::xcd::NfcEventReason reason;     //!< イベントの Reason (共用体の内部フォーマットを特定)

    union
    {
        nn::xcd::NfcErrorInfo    errorInfo;                  //!< エラー情報。 @a reason が @ref EventReason_Error のときに使用します。
        nn::nfc::TagInfo tagInfo;                //!< NFC タグの基本情報。 @reason が @ref EventReason_Detected のときに使用します。
        NtagData     ntagData;                   //!< NTAG シリーズのタグデータ。 @a reason が @ref EventReason_ReadFinish のときに使用します。
        PassThruData passThruData;               //!< パススルーモードの取得データ。 @a reason が @ref EventReason_PassThruResult のときに使用します。
        nn::xcd::NfcMifareData   mifareData;                 //!< MIFARE タグに対する操作への応答。 @a reason が @ref EventReason_MifareResult のときに使用します。

        char _dummy[nn::xcd::NfcReceiveDataSizeMax];         // 上限となるデータサイズを規定
    };
};

/**
 * @brief       タグ検出用のパラメータです。
 */
struct DiscoveryParameter
{
    nn::nfc::NfcProtocol protocolFilter; //!< 検出するタグのマスク
    uint16_t activationTimeout;     //!< polling のタイムアウト時間設定 (0 : 無制限, 1 - 65535 ms)
    uint16_t discoveryPeriod;       //!< polling の周期 (0 - 65535 ms)
};

/**
 * @brief       NTAG Read アクセス用のパラメータです。
 */
struct NtagReadParameter
{
    uint16_t  timeoutMsec;                //!< コマンドのタイムアウト (ミリ秒単位で指定)
    bool      isPasswordRequired;         //!< パスワード認証を行うかどうかのフラグ
    NN_PADDING1;
    nn::nfc::TagId tagId;                 //!< アクセスするタグの UID (All 0 の場合は指定なし）
    int32_t   blockCount;                 //!< Read コマンドで読み込むブロックの数
    nn::xcd::NtagReadAddress addresses[nn::xcd::NtagReadBlockCountMax];  //!< 各ブロックのデータ読み込みアドレスを指定。ブロック毎に 60 ページ分指定可能
};

/**
 * @brief       NTAG Write アクセス用のパラメータです。
 */
struct NtagWriteParameter
{
    uint16_t  timeoutMsec;            //!< タイムアウト (ミリ秒単位で指定)
    bool      isPasswordRequired;     //!< パスワード認証を行うかどうかのフラグ
    uint8_t   type2TagVersion;        //!< Type2 タグのバージョン
    nn::nfc::TagId tagId;             //!< アクセスするタグの UID
    nn::xcd::NtagWriteData ntagWriteData;      //!< タグに書き込むデータ
};

/**
 * @brief       パススルーコマンド用のデータです。
 */
struct PassThruParameter
{
    uint16_t  timeoutMsec;                       //!< タイムアウト (ミリ秒単位で指定)
    NN_PADDING2;
    nn::nfc::TagId tagId;                        //!< アクセスするタグの UID (すべて 0 の場合は最初に見つけたタグが対象)
    uint32_t  sendDataSize;                      //!< パススルーコマンドのデータサイズ
    char      sendData[nn::xcd::NfcPassThruDataSizeMax];     //!< 送信データ
};

/**
 * @brief       MIFARE Read アクセス用のパラメータです。
 */
struct MifareReadParameter
{
    uint16_t                 timeoutMsec;   //!< コマンドのタイムアウト (ミリ秒単位で指定)
    nn::xcd::MifareKeyValueFormat     keyFormat;     //!< 認証に使用する鍵の指定方法
    NN_PADDING1;
    nn::nfc::TagId           tagId;         //!< アクセスするタグの UID
    int32_t                  blockCount;    //!< 読み込むブロックの数 (1 - 16)
    nn::xcd::MifareReadBlockParameter blocks[nn::xcd::MifareReadBlockCountMax];   //!< 読み込むブロックの指定。blockCount 分の要素が有効
};

/**
 * @brief       MIFARE Write アクセス用のパラメータです。
 */
struct MifareWriteParameter
{
    uint16_t                  timeoutMsec;  //!< コマンドのタイムアウト (ミリ秒単位で指定)
    nn::xcd::MifareKeyValueFormat      keyFormat;    //!< 認証に使用する鍵の指定方法
    NN_PADDING1;
    nn::nfc::TagId            tagId;        //!< アクセスするタグの UID
    int32_t                   blockCount;   //!< 書き込むブロックの数 (1 - 16)
    nn::xcd::MifareWriteBlockParameter blocks[nn::xcd::MifareWriteBlockCountMax];   //!< 書き込むブロックの指定。blockCount 分の要素が有効
};
//--

/**
 * @brief   ライブラリの種別を表す列挙型です。
 */
enum LibraryType
{
    LibraryType_Nfc,           //!< NFC 機能 On/Off のみの NFC ライブラリ
    LibraryType_All,           //!< NFC 機能 On/Off + MIFARE + パススルーの NFC ライブラリ
    LibraryType_Nfp,           //!< NFP ライブラリ
    LibraryType_NfcMifare,     //!< MIFARE ライブラリ
};

/**
 * @brief   Nfc 処理がリセットさせられた要因を表す列挙型です。
 */
enum ResetReason
{
    ResetReason_NfcDisabled,             //!< NFC 機能が無効になりました
    ResetReason_Sleep,                   //!< 本体がスリープ状態になりました
    ResetReason_NotForeground,           //!< アプリケーションが最前面で動作していないため NFC 機能を利用できなくなりました。
    ResetReason_NfcDeviceNotFound        //!< 処理対象の NFC デバイスが見つからなくなりました。
};

struct ResetInfo
{
    ResetReason reason;        //!< リセット理由
};

}}}}  // namespace nn::nfc::server::core
