﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/** @file
    @brief      NFC に関する非公開 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>

namespace nn {
namespace nfc {

/**
 * @name    Private API
 * @{
 */

/**
 * @brief   NFC ライブラリをシステムモードで初期化します。
 *
 * @pre
 *  - GetState() == @ref State_None
 *
 * @post
 *  - NFC ライブラリがシステムモードで初期化済みの状態になります。
 *  - GetState() == @ref State_Init
 *
 * @details
 * nfc_PrivateApi.h で定義されたシステム用の API を使用するためには、
 * この API で NFC ライブラリを初期化する必要があります。@n
 * システムモードでは、一般公開 API も使用できます。
 *
 */
void InitializeSystem() NN_NOEXCEPT;

/**
 * @brief   NFC ライブラリを終了します。
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @post
 *  - GetState() == @ref State_None
 *
 * @details
 *  この関数は NFC ライブラリの使用を終える際に呼び出してください。
 *  InitializeSystem() で NFC ライブラリを初期化した場合には、この API でライブラリを終了します。
 *  必要に応じて @ref ReleasePassThroughSession() 、 @ref StopDetection() が内部で自動的に実行されます。
 */
void FinalizeSystem() NN_NOEXCEPT;

/**
 * @brief         NFC 機能が有効か否かを設定します。
 *
 * @param[in]     enable                      NFC 機能が有効か否かを表す値
 *
 * @pre
 *  - ライブラリが @ref InitializeSystem() にて初期化済みの状態である必要があります。
 *  - GetState() == @ref State_Init
 *
 * @details NFC 機能が有効か否かを設定します。
 */
void SetNfcEnabled(bool enable) NN_NOEXCEPT;

/**
 * @}
 */

}  // nfc
}  // nn
