﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/news_Types.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace news {

/*!
    @brief      ニュースの最大保存件数です。
*/
const int DataCountMax = 300;

/*!
    @brief      トピックの最大保存件数です。
*/
const int TopicCountMax = 1024;

/*!
    @brief      ニュースの保存ストレージのサイズです。
*/
const int64_t StorageSize = 100 * 1024 * 1024;

/*!
    @brief      パスフレーズの最大長です。
*/
const size_t PassphraseLengthMax = 127;

/*!
    @brief      トピック ID の最大長です。
*/
const size_t TopicIdLengthMax = 31;

/*!
    @brief      データ名の最大長です。
*/
const size_t DataNameLengthMax = 95;

/*!
    @brief      SQL のキーに指定できる文字列の最大長です。
*/
const size_t KeyLengthMax = 31;

/*!
    @brief      SQL の更新用文字列値に指定できる文字列の最大長です。
*/
const size_t StringValueLengthMax = 127;

/*!
    @brief      SQL の括弧のネストの最大深さです。
*/
const size_t NestDepthMax = 8;

/*!
    @brief      SQL の WHERE 句に指定できる文字列の最大長です。
*/
const size_t WherePhraseLengthMax = 1023;

/*!
    @brief      SQL の ORDER BY 句に指定できる文字列の最大長です。
*/
const size_t OrderByPhraseLengthMax = 1023;

/*!
    @brief      ニュースの購読状態です。

    @details
                本列挙子は廃止予定です。@n
                代わりに、 @ref SubscriptionStatus を使用してください。
*/
enum SubscribeStatus
{
    SubscribeStatus_Unconfigured   = 0, //!< 未設定。
    SubscribeStatus_Unsubscribed   = 1, //!< 未購読。
    SubscribeStatus_Subscribed     = 2, //!< 購読。
    SubscribeStatus_AutoSubscribed = 3  //!< 自動購読。
};

/*!
    @brief      ニュースの購読状態です。

    @details
                ニュースの種類によっては、ユーザーの確認を得ずに購読するものがあります。@n
                この場合、そのニュースの購読状況は「自動購読」になります。

                ただし、自動購読対象のニュースであっても、ユーザーが既に未購読の意思を表明している場合、自動購読は行いません。
*/
enum SubscriptionStatus
{
    SubscriptionStatus_Unconfigured   = 0, //!< 未設定。
    SubscriptionStatus_Unsubscribed   = 1, //!< 未購読。
    SubscriptionStatus_Subscribed     = 2, //!< 購読。
    SubscriptionStatus_AutoSubscribed = 3  //!< 自動購読。
};

/*!
    @brief      購読状態フィルターです。
*/
enum SubscriptionStatusFilter
{
    SubscriptionStatusFilter_None         = 0, //!< フィルターなし。
    SubscriptionStatusFilter_Unsubscribed = 1, //!< 未購読トピックでフィルターする。
    SubscriptionStatusFilter_Subscribed   = 2  //!< 購読、または、自動購読トピックでフィルターする。
};

/*!
    @brief      ニュースデータの ID です。
*/
struct NewsId
{
    char value[24];
};

/*!
    @brief      ユーザー ID です。

    @details
                個人宛ニュースの場合、有効な 10 進数文字列が設定されます。@n
                デバイス宛ニュースの場合、空文字列が設定されます。
*/
struct UserId
{
    char value[24];
};

/*!
    @brief      トピック ID です。
*/
struct TopicId
{
    char value[TopicIdLengthMax + 1]; //!< 値。
};

/*!
    @brief      ニュースデータベースのレコードです。
*/
struct NewsRecord
{
    NewsId newsId;                    //!< ニュース ID。
    UserId userId;                    //!< ユーザー ID。
    TopicId topicId;                  //!< トピック ID。
    nn::time::PosixTime receivedTime; //!< 受信時刻。
    nn::Bit8 _padding1[12];
    int32_t read;                     //!< 閲覧回数。0 の場合、未読。
    int32_t newly;                    //!< 新着フラグ。
    int32_t displayed;                //!< 表示実績。
    nn::Bit8 _padding2[8];
    int32_t extra1;                   //!< 拡張領域1。
    int32_t extra2;                   //!< 拡張領域2。
};

/*!
    @brief      ニュースデータベースのレコードです。（バージョン 1）

    @note
                バイナリ互換性のために残しています。
*/
struct NewsRecordV1
{
    NewsId newsId;                    //!< ニュース ID。
    UserId userId;                    //!< ユーザー ID。
    nn::time::PosixTime receivedTime; //!< 受信時刻。
    int32_t read;                     //!< 閲覧回数。0 の場合、未読。
    int32_t newly;                    //!< 新着フラグ。
    int32_t displayed;                //!< 表示実績。
    int32_t extra1;                   //!< 拡張領域1。
};

}}
