﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/news_ServiceHeap.h>
#include <nn/news/detail/ipc/news_ServiceConfig.h>

namespace nn { namespace news { namespace detail { namespace service {

/*!
    @brief      サービスのメモリ管理モジュールです。
*/
class ServiceMemoryManager
{
public:
    /*!
        @brief      NewsService に割り当てるバッファサイズです。
    */
    static const size_t NewsServiceSize = 256;

    /*!
        @brief      NewsService のヒープサイズです。
    */
    static const size_t NewsServiceHeapSize = NewsServiceSize * detail::ipc::NewsServiceCountMax;

    /*!
        @brief      NewlyArrivedEventHolder に割り当てるバッファサイズです。
    */
    static const size_t NewlyArrivedEventHolderSize = 128;

    /*!
        @brief      NewlyArrivedEventHolder のヒープサイズです。
    */
    static const size_t NewlyArrivedEventHolderHeapSize = NewlyArrivedEventHolderSize * detail::ipc::NewlyArrivedEventHolderCountMax;

    /*!
        @brief      OverwriteEventHolder に割り当てるバッファサイズです。
    */
    static const size_t OverwriteEventHolderSize = 128;

    /*!
        @brief      OverwriteEventHolder のヒープサイズです。
    */
    static const size_t OverwriteEventHolderHeapSize = OverwriteEventHolderSize * detail::ipc::OverwriteEventHolderCountMax;

    /*!
        @brief      NewsDataService に割り当てるバッファサイズです。
    */
    static const size_t NewsDataServiceSize = 256;

    /*!
        @brief      NewsDataService のヒープサイズです。
    */
    static const size_t NewsDataServiceHeapSize = NewsDataServiceSize * detail::ipc::NewsDataServiceCountMax;

    /*!
        @brief      NewsDatabaseService に割り当てるバッファサイズです。
    */
    static const size_t NewsDatabaseServiceSize = 256;

    /*!
        @brief      NewsDatabaseService のヒープサイズです。
    */
    static const size_t NewsDatabaseServiceHeapSize = NewsDatabaseServiceSize * detail::ipc::NewsDatabaseServiceCountMax;

private:
    /*!
        @brief      コンストラクタです。
    */
    ServiceMemoryManager() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~ServiceMemoryManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ServiceMemoryManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ServiceMemoryManager, instance);
        return instance;
    }

public:
    /*!
        @brief      NewsService 用アロケータを取得します。

        @return     NewsService 用アロケータ。
    */
    nn::sf::ExpHeapAllocator& GetNewsServiceAllocator() NN_NOEXCEPT;

    /*!
        @brief      NewlyArrivedEventHolder 用アロケータを取得します。

        @return     NewlyArrivedEventHolder 用アロケータ。
    */
    nn::sf::ExpHeapAllocator& GetNewlyArrivedEventHolderAllocator() NN_NOEXCEPT;

    /*!
        @brief      OverwriteEventHolder 用アロケータを取得します。

        @return     OverwriteEventHolder 用アロケータ。
    */
    nn::sf::ExpHeapAllocator& GetOverwriteEventHolderAllocator() NN_NOEXCEPT;

    /*!
        @brief      NewsDataService 用アロケータを取得します。

        @return     NewsDataService 用アロケータ。
    */
    nn::sf::ExpHeapAllocator& GetNewsDataServiceAllocator() NN_NOEXCEPT;

    /*!
        @brief      NewsDatabaseService 用アロケータを取得します。

        @return     NewsDatabaseService 用アロケータ。
    */
    nn::sf::ExpHeapAllocator& GetNewsDatabaseServiceAllocator() NN_NOEXCEPT;

private:
    //
    ServiceHeap<NewsServiceHeapSize> m_NewsServiceHeap;
    ServiceHeap<NewlyArrivedEventHolderHeapSize> m_NewlyArrivedEventHolderHeap;
    ServiceHeap<OverwriteEventHolderHeapSize> m_OverwriteEventHolderHeap;
    ServiceHeap<NewsDataServiceHeapSize> m_NewsDataServiceHeap;
    ServiceHeap<NewsDatabaseServiceHeapSize> m_NewsDatabaseServiceHeap;
};

}}}}
