﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>
#include <curl/curl.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      トピックリストのダウンロードを行うモジュールです。
*/
class TopicListDownloader
{
public:
    /*!
        @brief      トピックリストデータの最大サイズです。
    */
    static const size_t TopicListDataSizeMax = 1024;

public:
    /*!
        @brief      コンストラクタです。
    */
    TopicListDownloader() NN_NOEXCEPT;

    /*!
        @brief      キャンセルイベントを設定します。

        @param[in]  pCancelEvent    キャンセルイベント。

        @pre
            - pCancelEvent != nullptr
    */
    void SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT;

    /*!
        @brief      ダウンロード処理を行います。

        @param[out] outCount        取得したトピック ID 数。
        @param[out] outTopicIds     トピック ID リスト。
        @param[in]  count           トピック ID リストの要素数。
        @param[in]  targetAppId     ターゲットアプリケーションのアプリケーション ID。
        @param[in]  newsAppId       ニュースアプリケーションのアプリケーション ID。
        @param[in]  newsPassphrase  ニュースアプリケーションのパスフレーズ。
        @param[in]  pConnection     ネットワーク接続要求オブジェクト。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outTopicIds != nullptr
            - count > 0
            - targetAppId != nn::ApplicationId::GetInvalidId()
            - newsAppId != nn::ApplicationId::GetInvalidId()
            - newsPassphrase != nullptr
            - pConnection != nullptr
    */
    nn::Result Download(int* outCount, TopicId* outTopicIds, int count,
        nn::ApplicationId targetAppId, nn::ApplicationId newsAppId, const char* newsPassphrase,
        nn::nifm::NetworkConnection* pConnection) NN_NOEXCEPT;

private:
    //
    nn::os::Event* m_pCancelEvent;
    //
    Bit8 m_ArchiveData[TopicListDataSizeMax];
    size_t m_ArchiveSize;
    Bit8 m_DecodedData[TopicListDataSizeMax];
    size_t m_DecodedSize;

private:
    //
    nn::Result StepDownload(nn::ApplicationId appId, nn::nifm::NetworkConnection* pConnection) NN_NOEXCEPT;
    nn::Result StepDecode(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT;
    nn::Result StepRead(int* outCount, TopicId* outTopicIds, int count) NN_NOEXCEPT;
    //
    bool IsCanceled() NN_NOEXCEPT;

private:
    //
    static CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT;
    static size_t HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT;
    static int HttpProgressFunction(void* param, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT;
};

}}}}}
