﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      One2One 通知の管理モジュールです。
*/
class One2OneNotificationManager
{
public:
    /*!
        @brief      記録する通知の最大件数です。
    */
    static const int NotificationCountMax = 30;

    /*!
        @brief      通知レコードです。
    */
    struct Record
    {
        // 8
        nn::time::PosixTime receivedTime;
        // 8
        uint64_t userId; // 将来ユーザー情報を残す可能性があるので予約しておく。
        // 256
        Url url;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 272);

    /*!
        @brief      システム起動中のみ使用される揮発性のレコードです。
    */
    struct VolatileRecord
    {
        // 8
        nn::os::Tick runnableTick;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    One2OneNotificationManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static One2OneNotificationManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(One2OneNotificationManager, instance);
        return instance;
    }

public:
    /*!
        @brief      通知リストを読み込みます。

        @return     処理結果。
    */
    nn::Result Load() NN_NOEXCEPT;

    /*!
        @brief      通知リストをクリアします。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      現在実行すべき通知を取得します。

        @param[in]  outRecord   レコード。

        @return     処理結果。

        @pre
            - outRecord != nullptr

        @details
                    本関数は、実行状態の通知を優先して取得します。@n
                    実行状態の通知が存在しない場合、実行可能状態の通知を 1 つ選択します。

                    現在実行すべき通知がない場合、 ResultNotFound を返します。
    */
    nn::Result GetCurrentNotification(Record* outRecord) const NN_NOEXCEPT;

    /*!
        @brief      次回のスケジュールまでの時間間隔を取得します。

        @param[out] outInterval 次回のスケジュールまでの時間間隔。

        @return     スケジュールすべき通知が存在するかどうか。

        @pre
            - outInterval != nullptr
    */
    bool GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;

    /*!
        @brief      処理を行うべき時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::TimerEvent& GetEvent() NN_NOEXCEPT;

    /*!
        @brief      プッシュ通知の受信を通知します。

        @param[in]  url         配信リスト URL。
        @param[in]  waitTime    待機時間。

        @return     処理結果。

        @details
                    プッシュ通知トリガーによる実行では、多数台同時リクエストが発生する可能性があります。@n
                    負荷分散のため、タスクの実行はデバイスごとにランダムな待機時間を設定してください。
    */
    nn::Result NotifyNotificationReceived(const Url& url, int32_t waitTime = 0) NN_NOEXCEPT;

    /*!
        @brief      通知の実行完了を通知します。

        @param[in]  record  レコード。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyDone(const Record& record, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      ネットワークの接続を通知します。
    */
    void NotifyNetworkConnected() NN_NOEXCEPT;

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::os::TimerEvent m_Event;
    //
    Record m_Records[NotificationCountMax];
    VolatileRecord m_VolatileRecords[NotificationCountMax];
    int m_Count;
    //
    bool m_IsNetworkConnectionAvailable;

private:
    //
    nn::Result LoadImpl() NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    bool Verify() const NN_NOEXCEPT;
    //
    int SearchRecord(const Url& url) const NN_NOEXCEPT;
    //
    void AddRecord(const Url& url) NN_NOEXCEPT;
    void RemoveRecord(int index) NN_NOEXCEPT;
    void RemoveOldestRecord() NN_NOEXCEPT;
    //
    void SetWait(int index, int32_t waitTime) NN_NOEXCEPT;
    //
    void SetScheduleTimer() NN_NOEXCEPT;
    bool GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;

private:
    //
    static bool IsServerTemporaryError(nn::Result result) NN_NOEXCEPT;
    static bool IsServerFailureError(nn::Result result) NN_NOEXCEPT;
    //
    static bool VerifyUrl(const Url& url) NN_NOEXCEPT;
};

}}}}}
