﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>
#include <nn/news/detail/service/msgpack/news_MsgpackEventHandler.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      ニュースリストのリーダーです。
*/
class NewsListReader : public detail::service::msgpack::MsgpackEventHandler
{
private:
    NN_DISALLOW_COPY(NewsListReader);
    NN_DISALLOW_MOVE(NewsListReader);

public:
    /*!
        @brief      リスト情報です。
    */
    struct ListInfo
    {
        TopicId topicId;
        bool isInService;
        bool isNintendoAccountRequired;
        bool isTestDistribution;
    };

    /*!
        @brief      データエントリーのパース結果です。
    */
    struct DataEntryParseBits
    {
        Bit64 formatVersion : 1;
        Bit64 semanticsVersion : 1;
        Bit64 newsId : 1;
        Bit64 dataId : 1;
        Bit64 url : 1;
        Bit64 size : 1;
        Bit64 isOverwrite : 1;
        Bit64 defaultLanguage : 1;
        Bit64 language : 1;
        Bit64 defaultLanguageEntry : 1;
        Bit64 conditions : 1;
    };

    /*!
        @brief      リスト情報のパース結果です。
    */
    struct ListInfoParseBits
    {
        Bit64 topicId : 1;
        Bit64 isInService : 1;
        Bit64 isNintendoAccountRequired : 1;
        Bit64 isTestDistribution : 1;
    };

    /*!
        @brief      受信条件です。
    */
    struct ReceptionConditions
    {
        bool isSerialBits;
        NN_PADDING1;
        nn::Bit16 serialBits;
    };

    /*!
        @brief      データエントリーです。
    */
    struct DataEntry
    {
        NewsVersion version;
        NN_PADDING6;
        uint64_t newsId;
        char defaultLanguage[8];
        char language[8];
        uint64_t dataId;
        Url url;
        uint64_t size;
        bool isOverwrite;
        NN_PADDING3;
        ReceptionConditions conditions;
    };

public:
    /*!
        @brief      コンストラクタです。
    */
    NewsListReader() NN_NOEXCEPT;

    /*!
        @brief      言語を設定します。

        @param[in]  language    言語。

        @return     処理結果。

        @pre
            - language != nullptr

        @details
                    ニュースデータは言語ごとにデータが分かれています。（内容は同じ）@n
                    本関数で言語を設定することで、設定した言語のニュースデータを取得するためのエントリーが取得できるようになります。

                    言語を設定しない場合、デフォルト設定（language=default）のエントリーのみを取得します。
    */
    void SetLanguage(const char* language) NN_NOEXCEPT;

    /*!
        @brief      ニュースリストを読み込みます。

        @param[out] outCount    取得したエントリー数。
        @param[in]  stream      入力ストリーム。
        @param[in]  entries     エントリーバッファ。
        @param[in]  offset      オフセット。
        @param[in]  count       エントリーバッファの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr

        @details
                    エントリーバッファの要素数を 0 にした場合、リスト情報の取得のみを行います。
    */
    nn::Result Read(int* outCount, nne::nlib::InputStream& stream, DataEntry* entries, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      リスト情報を取得します。

        @return     リスト情報。
    */
    const ListInfo& GetListInfo() const NN_NOEXCEPT;

private:
    //
    detail::service::msgpack::JsonPath m_JsonPath;
    //
    ListInfo m_ListInfo;
    ListInfoParseBits m_ListInfoBits;
    //
    DataEntry m_Entry;
    DataEntryParseBits m_EntryBits;
    //
    DataEntry* m_Entries;
    int m_Index;
    int m_Offset;
    int m_Count;
    //
    const char* m_CurrentLanguage;
    //
    bool m_IsLanguageEntryParsing;
    bool m_IsConditionsParsing;

private:
    // MsgpackEventHandler
    virtual bool OnStartObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnEndObject(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnStartArray(const detail::service::msgpack::JsonPath& jsonPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnBoolean(const detail::service::msgpack::JsonPath& jsonPath, bool value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnInteger(const detail::service::msgpack::JsonPath& jsonPath, const detail::service::msgpack::AnyInteger& value) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool OnString(const detail::service::msgpack::JsonPath& jsonPath, const char* value, size_t length, bool isOverflowed) NN_NOEXCEPT NN_OVERRIDE;
};

}}}}}
