﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      フォアグラウンドからストレージにアクセスする際のロック管理を行うモジュールです。
*/
class ForegroundStorageLockManager
{
private:
    NN_DISALLOW_COPY(ForegroundStorageLockManager);
    NN_DISALLOW_MOVE(ForegroundStorageLockManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    ForegroundStorageLockManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static ForegroundStorageLockManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ForegroundStorageLockManager, instance);
        return instance;
    }

public:
    /*!
        @brief      データアクセサ用のロックを取得します。

        @details
                    データアクセサ用のロックは連続で取得可能です。@n
                    連続で取得した回数だけ、ロックを解放する必要があります。

                    インポーター用のロックが取得されている場合、ロックが解除されるまでブロックします。
    */
    void AcquireDataAccessorLock() NN_NOEXCEPT;

    /*!
        @brief      データアクセサ用のロックを解放します。

        @details
                    複数回ロックが取得されていた場合、取得回数分だけ本関数を呼び出す必要があります。
    */
    void ReleaseDataAccessorLock() NN_NOEXCEPT;

    /*!
        @brief      インポーター用のロックを取得します。

        @return     ロックが取得できたかどうか。

        @details
                    インポーター用のロックは連続で取得することはできません。@n
                    また、データアクセサ用のロックが取得されていた場合、本関数は失敗します。
    */
    bool TryAcquireImporterLock() NN_NOEXCEPT;

    /*!
        @brief      インポーター用のロックを解放します。
    */
    void ReleaseImporterLock() NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    nn::os::ReaderWriterLock m_AccessorsImporterLock;
};

}}}}}
