﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/core/news_FileSystemConfig.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      ファイルシステムライブラリのラッパーモジュールです。
*/
class FileSystem
{
public:
    /*!
        @brief      テストモードを有効にします。

        @details
                    テストモードが有効な場合、以下の API はストレージへのアクセスを行いません。

                    - FileSystem::Mount
                    - FileSystem::Unmount
                    - FileSystem::Commit
                    - FileSystem::Rollback

                    テスト実行前に必要とするストレージをマウントしてください。
    */
    static void EnableTestMode() NN_NOEXCEPT;

    /*!
        @brief      ストレージをマウントします。

        @param[in]  mountName   マウント名。
        @param[in]  isLock      ストレージをロックするかどうか。

        @return     処理結果。
    */
    static nn::Result Mount(const char* mountName, bool isLock = true) NN_NOEXCEPT;

    /*!
        @brief      ストレージをアンマウントします。

        @param[in]  mountName   マウント名。
        @param[in]  isLocked    ストレージをロックしていたかどうか。
    */
    static void Unmount(const char* mountName, bool isLocked = true) NN_NOEXCEPT;

    /*!
        @brief      ストレージをロックします。

        @param[in]  mountName   マウント名。
    */
    static void Lock(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージのロックを解除します。

        @param[in]  mountName   マウント名。
    */
    static void Unlock(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをコミットします。

        @param[in]  mountName   マウント名。

        @return     処理結果。

        @details
                    ストレージに書き込んだファイルを確定するためには、コミットが必要です。
    */
    static nn::Result Commit(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージをロールバックします。

        @param[in]  mountName   マウント名。
    */
    static void Rollback(const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ストレージの空き容量を取得します。

        @param[out] outSize     空き容量。
        @param[in]  mountName   マウント名。

        @return     処理結果。

        @pre
            - outSize != nullptr
    */
    static nn::Result GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に作成します。

        @param[in]  path        パス。
        @param[in]  isFilePath  ファイルパスを指定したかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    ファイルパスを指定した場合、親ディレクトリの生成を行います。
    */
    static nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath = false) NN_NOEXCEPT;

    /*!
        @brief      ディレクトリを再帰的に削除します。

        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - path != nullptr
    */
    static nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;

    /*!
        @brief      ファイルを新規作成します。

        @param[in]  path            パス。
        @param[in]  size            ファイルサイズ。
        @param[in]  ignoreIfExists  ファイルがすでに存在する時に成功を返すかどうか。

        @return     処理結果。

        @pre
            - path != nullptr

        @details
                    親ディレクトリが存在しない場合、親ディレクトリも作成します。
    */
    static nn::Result CreateFile(const char* path, int64_t size, bool ignoreIfExists = true) NN_NOEXCEPT;

    /*!
        @brief      ファイルサイズを取得します。

        @param[out] outSize ファイルサイズ。
        @param[in]  path    パス。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - path != nullptr
    */
    static nn::Result GetFileSize(int64_t* outSize, const char* path) NN_NOEXCEPT;
};

}}}}}

/*!
    @brief      データ保存ストレージのマウント名です。
*/
#define NN_DETAIL_NEWS_DATA_MOUNT_NAME "news"

/*!
    @brief      システム情報ストレージのマウント名です。
*/
#define NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME "news-sys"

/*!
    @brief      ダウンロードストレージのマウント名です。
*/
#define NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME "news-dl"

/*!
    @brief      ストレージを局所的にマウントします。

    @details
                各モジュールのロックとストレージのロックが競合してデッドロックが発生しないよう注意してください。
*/
#define NN_DETAIL_NEWS_STORAGE_SCOPED_MOUNT(mountName) \
    detail::service::core::FileSystem::Mount(mountName, true);          \
    NN_UTIL_SCOPE_EXIT                                                  \
    {                                                                   \
        detail::service::core::FileSystem::Unmount(mountName, true);    \
    }

/*!
    @brief      ストレージを局所的にロックします。

    @details
                各モジュールのロックとストレージのロックが競合してデッドロックが発生しないよう注意してください。
*/
#define NN_DETAIL_NEWS_STORAGE_SCOPED_LOCK(mountName) \
    detail::service::core::FileSystem::Lock(mountName);         \
    NN_UTIL_SCOPE_EXIT                                          \
    {                                                           \
        detail::service::core::FileSystem::Unlock(mountName);   \
    }

#if NN_DETAIL_NEWS_SYSTEM_STORAGE_MOUNT_ALWAYS == 1

/*!
    @brief      システム情報ストレージを局所的にマウントします。
*/
#define NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT() \
    NN_DETAIL_NEWS_STORAGE_SCOPED_LOCK(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME)

#else

/*!
    @brief      システム情報ストレージを局所的にマウントします。
*/
#define NN_DETAIL_NEWS_SYSTEM_STORAGE_SCOPED_MOUNT() \
    NN_DETAIL_NEWS_STORAGE_SCOPED_MOUNT(NN_DETAIL_NEWS_SYSTEM_MOUNT_NAME)

#endif

/*!
    @brief      ダウンロードストレージを局所的にマウントします。
*/
#define NN_DETAIL_NEWS_DOWNLOAD_STORAGE_SCOPED_MOUNT() \
    NN_DETAIL_NEWS_STORAGE_SCOPED_MOUNT(NN_DETAIL_NEWS_DOWNLOAD_MOUNT_NAME)
