﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief nn::netdiag の型定義(非公開)
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace netdiag {

    // NATタイプ判定サーバ
    const char ServerNameNncs1[] = "nncs1-%.n.n.srv.nintendo.net"; //!< NAT タイプ判定サーバ1
    const char ServerNameNncs2[] = "nncs2-%.n.n.srv.nintendo.net"; //!< NAT タイプ判定サーバ2

    /**
     * @brief   NAT タイプ判定サーバを表す値です。
     */
    enum NncsServer
    {
        NncsServer_1 = 0,
        NncsServer_2 = 1
    };

    /**
     * @brief   NAT タイプ判定サーバに送信するリクエストパケットフォーマットです。
     */
    struct RequestPacket
    {
        uint32_t testId;           // !< テスト識別子
        uint32_t PerceivedPort;    // !< 0固定
        uint32_t PerveivedAddress; // !< 0固定
        uint32_t SourceAddress;    // !< 0固定
    };

    /**
     * @brief   NAT タイプ判定サーバから受信するレスポンスパケットフォーマットです。
     */
    struct ResponsePacket
    {
        uint32_t testId;           // !< テスト識別子(リクエストと同じとなる)
        uint32_t PerceivedPort;    // !< クライアントの外部ポート
        uint32_t PerveivedAddress; // !< クライアントの外部アドレス
        uint32_t SourceAddress;    // !< NAT タイプ判定サーバの内部アドレス( NAT タイプ判定では使用しない)
    };

    /**
     * @brief   リクエストパケットと、レスポンスパケットに含まれるテスト識別子です。
     */
    enum TestId
    {
        TestId_Dummy       = 0,   // !< レスポンスを求めない場合
        TestId_Simplified1 = 101, // !< マッピングタイプ判定用
        TestId_Simplified2 = 102, // !< フィルタリングタイプ判定用
        TestId_Simplified3 = 103  // !< マッピングルール判定用
    };

    // リトライ回数
    const int RetryTestTime = 3;

//----------------------------------------------------------------
//
    /**
     * @brief   NAT タイプ判定に利用する、マッピングタイプです。
     */
    enum MappingType
    {
        MappingType_Eim = 0,  // Endpoiont Independent Mapping
        MappingType_Edm = 1   // Endpoint Dependent Mapping
    };
    /**
     * @brief   NAT タイプ判定に利用する、フィルタリングタイプです。
     */
    enum FilteringType
    {
        FilteringType_Pif = 0, // Port Independent Filtering
        FilteringType_Pdf = 1  // Port Dependent Filtering
    };

//----------------------------------------------------------------
// レスポンス解析

    /**
     * @brief   レスポンス解析用の、パケット種類を表す値です。
     */
    enum SimplifiedIndex
    {
        SimplifiedIndex_1 = 0,
        SimplifiedIndex_2 = 1,
        SimplifiedIndex_3 = 2
    };

    /**
     * @brief   レスポンス解析用の結果です。
     */
    enum ResponseResult
    {
        ResponseResult_Finished        = 0,  // すべて揃って完了
        ResponseResult_ReceivedMapping = 1,  // フィルタリング判定用はまだでマッピング関連は揃った
        ResponseResult_OnTheWay        = 2,  // まだ途中
        ResponseResult_Unrelated       = 3   // 関係ないパケットだった
    };

    /**
     * @brief   解析結果のレスポンス受信フラグの値です。
     */
    enum ResponseReceived
    {
        ResponseReceived_Simplified1 = (1<<0), // マッピングタイプ判定用
        ResponseReceived_Simplified2 = (1<<1), // フィルタリングタイプ判定用
        ResponseReceived_Simplified3 = (1<<2), // マッピングルール判定用
        ResponseReceived_MappingSimplified = ResponseReceived_Simplified1 | ResponseReceived_Simplified3,
        ResponseReceived_AllSimplified = ResponseReceived_Simplified1 | ResponseReceived_Simplified2 | ResponseReceived_Simplified3,
    };

    /**
     * @brief   解析結果を格納する構造体です。
     */
    struct ResponseStat
    {
        // 送信ポート
        uint16_t      sourcePort;         // リクエスト送信のソースポート

        // レスポンス格納
        uint32_t      receivedFlag;       // レスポンス受信フラグ
        uint16_t      port[3];            // レスポンス受信元ポート

        // レスポンスからの解析場所
        bool          isSuccess;          // 成功したか
        MappingType   mappingType;        // マッピングタイプ
        FilteringType filteringType;      // フィルタリングタイプ
        uint16_t      portDifference;     // ポート差分
    };

}} // nn::netdiag
