﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   LBL ライブラリのBL制御機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/lbl/lbl_Type.h>


namespace nn {
namespace lbl {

/**
 * @brief 自動輝度調整を有効にします。
 *
 * @pre         初期化されている必要があります。
 * @post        自動輝度調整が有効になります。
 * @details     自動輝度調整を有効にします。
 */
void EnableAutoBrightnessControl() NN_NOEXCEPT;

/**
 * @brief 自動輝度調整を無効にします。
 *
 * @pre         初期化されている必要があります。
 * @post        自動輝度調整が無効になります。
 * @details     自動輝度調整を無効にします。
 */
void DisableAutoBrightnessControl() NN_NOEXCEPT;

/**
 * @brief 自動輝度調整の有効/無効の状態を取得します。
 *
 * @return      自動輝度調整の有効/無効の状態を返します(true: 有効, false: 無効)。
 * @pre         初期化されている必要があります。
 * @details     自動輝度調整の有効/無効の状態を取得します。
 */
bool IsAutoBrightnessControlEnabled() NN_NOEXCEPT;

/**
 * @brief 明るさセンサーの値を設定します。
 *
 * @param[in]   sensorValue         明るさセンサーの値
 * @pre         初期化されている必要があります。
 * @post        @a sensorValue が現在使用する明るさセンサーの値になります。値の単位は lux です。
 * @details     明るさセンサーの値を設定します。
 *              自動輝度調整有効時に限って、明るさセンサーの値と輝度設定値からLCDバックライトの明るさが決定します。
 */
void SetAmbientLightSensorValue(float sensorValue) NN_NOEXCEPT;

/**
 * @brief 現在設定されている明るさセンサーの値を取得します。
 *
 * @return      現在設定されている明るさセンサーの値を返します。値の単位は lux です。
 * @param[out]  pOutOverflowAlert    光量過多のため内部の計算でオーバーフローが発生した場合に true をセットします。それ以外で false をセットします。
 * @pre         初期化されている必要があります。
 * @details     現在設定されている明るさセンサーの値を取得します。
 */
float GetAmbientLightSensorValue(bool* pOutOverflowAlert) NN_NOEXCEPT;

/**
 * @brief 明るさセンサーが使用可能かどうかを取得します。
 *
 * @return      明るさセンサーが使用可能かどうかを返します (true: 使用可能, false: 使用不可)。
 * @pre         初期化されている必要があります。
 * @details     明るさセンサーが使用可能かどうかを取得します。
 *              明るさセンサーを搭載していないハードウェア上で呼び出した場合、 false が返ります。
 */
bool IsAmbientLightSensorAvailable() NN_NOEXCEPT;

/**
 * @brief 自動輝度調整有効時のLCDバックライトの明るさの遷移に遅延を持たせます。
 * @deprecated この関数は廃止されました。呼び出しても効果はありません。将来のバージョンで削除予定です。
 */
NN_DEPRECATED void SetBrightnessReflectionDelayLevel(BrightnessChangeDirection direction, float delayLevel) NN_NOEXCEPT;

/**
 * @brief 自動輝度調整有効時のLCDバックライトの明るさの遷移に持たせている遅延レベルを返します。
 * @deprecated この関数は廃止されました。呼び出しても効果はありません（常に 0.0f が返ります）。将来のバージョンで削除予定です。
 */
NN_DEPRECATED float GetBrightnessReflectionDelayLevel(BrightnessChangeDirection direction) NN_NOEXCEPT;

} // lbl
} // nn


