﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットの共通引数の受け取りをおこなうクラスの非公開ヘッダ
 */

#pragma once

#include <nn/la/la_CommonArguments.h>
#include <nn/ae/ae.h>


namespace nn { namespace la {

/**
 * @brief   ライブラリアプレットの共通引数の受け取りをおこなうクラスです。
 *          これを利用するアプレットは、 libnn_applet.a, libnn_la_callee.a, libnn_ae.a をリンクする必要があります。
 */
class CommonArgumentsReader
{
private:
    bool m_IsInitialized;
    size_t m_CommonSizeInStorage;

    detail::CommonArgumentsDataLatest  m_CommonArgumentsData;

private:
    /**
     * @brief   ストレージから後方互換を保ちながら CommonArguments を読み出します。
     *
     * @param[in]  header        共通データのヘッダです。
     * @param[in]  storageHandle 共通データを読み出すためのストレージハンドルです。
     * @param[in]  storageSize   共通データ用のストレージサイズです。
     *
     * @return データ壊れを検出した場合は false を返します。
     */
    template <typename TCommonArgument>
    bool ReadCommonDataFromStorage(const detail::CommonArgumentsHeader& header, nn::ae::StorageHandle storageHandle, size_t storageSize) NN_NOEXCEPT;

    /**
     * @brief   古いバージョンの CommonArguments を最新版に更新して m_CommonArgumentsData に入れます。
     *
     * @param[in/out]  body        共通データのヘッダです。
     */
    template < typename TCommonArgument >
    void UpdateCommonDataToLatest( TCommonArgument& oldVersion ) NN_NOEXCEPT;

public:
    /**
     * @brief   コンストラクタです。
     *
     * @details
     */
    CommonArgumentsReader() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_CommonSizeInStorage(0)
    {
        m_CommonArgumentsData.laVersion = 0;
        m_CommonArgumentsData.colorIndex = -1;
        m_CommonArgumentsData.isPlayStartupSound = false;
        m_CommonArgumentsData.callerTimeStamp = 0;

        // 旧バージョンになかったパラメータは、コンストラクタで埋めたデフォルト値が用いられる
    }

    /**
     * @brief   デストラクタです。
     *
     * @details
     */
    ~CommonArgumentsReader() NN_NOEXCEPT
    {
    }

    /**
     * @brief   入力チャンネルから共通引数の pop を試みます。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *  取得に失敗した場合でも、入力チャンネルの状態に副作用を与えることがあります。
     *  内部で ReadFromStorage も呼び出します。
     */
    bool TryPopFromInChannel() NN_NOEXCEPT;

    /**
     * @brief   入力ストレージから実際にデータを読みます。
     *
     * @param[in] storageHandle 共通データを読み出すストレージのハンドルです。
     *
     * @return  取得に成功すれば true が、失敗すれば false が返ります。
     *
     * @details
     *    Storage の Pop を外部でおこないたい場合にはこちらを使用します。
     */
    bool ReadFromStorage( nn::ae::StorageHandle storageHandle ) NN_NOEXCEPT;

    /**
     * @brief   Storage の中の共通データのサイズを返します。
     *
     * @return  Storage の中の CommonArguments と CommonArgumentsHeader の実サイズを返します。
     *
     * @details
     *    LA 単体起動時に直接値を設定する用の API
     */
    size_t  GetCommonSize() const NN_NOEXCEPT;

    /**
     * @brief   呼び出し元が想定するライブラリアプレットのバージョンを取得します。
     *
     * @return  呼び出し元が想定するライブラリアプレットのバージョンが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    uint32_t GetLaVersion() const NN_NOEXCEPT;

    /**
     * @brief   呼び出し元が想定するライブラリアプレットのメジャーバージョンを取得します。
     *
     * @return  呼び出し元が想定するライブラリアプレットのメジャーバージョンが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    uint16_t GetLaVersionMajor() const NN_NOEXCEPT;

    /**
     * @brief   呼び出し元が想定するライブラリアプレットのマイナーバージョンを取得します。
     *
     * @return  呼び出し元が想定するライブラリアプレットのマイナーバージョンが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    uint16_t GetLaVersionMinor() const NN_NOEXCEPT;

    /**
     * @brief   色セット番号を取得します。
     *
     * @return  色セット番号が返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     *  呼び出し元で nn::oe::SetColorIndex() によって指定された色セット番号が取得できます。
     */
    int32_t GetColorIndex() const NN_NOEXCEPT;

    /**
     * @brief   起動時にサウンドを再生する必要があるかどうかを取得します。
     *
     * @return  起動時にサウンドを再生する必要があるかどうかが返ります。
     *
     * @pre     事前に TryPopFromInChannel() によって入力チャンネルから共通引数を pop する必要があります。
     *
     * @details
     */
    bool IsPlayStartupSound() const NN_NOEXCEPT;


    /**
     * @brief   呼び出し元のタイムスタンプの情報を取得します。
     *
     * @return  呼び出し元が記録した Tick 値を返します。
     *
     * @details
     *   アプレット起動時間計測用です。
     */
    nn::os::Tick GetCallerTimeStamp() const NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   共通引数をバッファからインポートします。
     *
     * @details
     *  この関数は、プロセス内で構築した共通引数を、疑似的に LA の引数受け取りに引き渡すために使用します。
     */
    void Import(const void* buffer, size_t size) NN_NOEXCEPT;

};

}} // namespace nn::la

