﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <curl/curl.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/http/json/http_RapidJsonInputStream.h>
#include <nn/http/stream/http_WebApiAccessorBase.h>

namespace nn { namespace http { namespace stream {

class CurlInputStreamBase
    : http::stream::WebApiAccessorBase
{
private:
    bool m_Opened {false};
    Result m_IoResult {ResultSuccess()};

    CURLM* m_CurlMulti {nullptr};

    struct
    {
        std::atomic<bool> empty;
        void* address;
        size_t total;
        size_t used;
    } m_BufferInfo {{false}, nullptr, 0u, 0u};
    size_t m_InputOffset {0u};

    static size_t HeaderFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;
    static size_t WriteFunction(char* ptr, size_t unitBytes, size_t count, void* pContext) NN_NOEXCEPT;

    Result FinalizeImpl() NN_NOEXCEPT;
    Result ExtractMultiPerformError(CURLM* curlMulti) NN_NOEXCEPT;

public:
    /**
        @brief コンストラクタ
        @param[in] curlHandle
            内部の通信処理に使用する libcURL のハンドル。 コンストラクタ内で curl_easy_reset() される。
        @param[in] pCancelable
            パース処理を途中で中止する可能性があれば、その中止を検知するための nn::util::Cancelable オブジェクトを指定します。
            パース処理中の特定のタイミングで中止されたかどうかが検査されます。検査の間隔は一定ではありません。
     */
    CurlInputStreamBase(CURL* curlHandle, const util::Cancelable* pCancelable = nullptr) NN_NOEXCEPT;
    ~CurlInputStreamBase() NN_NOEXCEPT;

    Result Initialize() NN_NOEXCEPT;

    using WebApiAccessorBase::SetDebugMode;

    using WebApiAccessorBase::SetSslContextHandler;
    using WebApiAccessorBase::SetUrl;
    using WebApiAccessorBase::SetHeaders;
    using WebApiAccessorBase::SetHttpDelete;
    using WebApiAccessorBase::SetHttpPatch;
    using WebApiAccessorBase::SetHttpPost;
    using WebApiAccessorBase::SetHttpPut;
    using WebApiAccessorBase::SetTimeoutSeconds;

    using WebApiAccessorBase::SetErrorHandler;
    using WebApiAccessorBase::SetUserAgent;

protected:
    /**
        @brief ストリームを開きます。
        @details
        ストリームを開き、 ImportJsonByRapidJson() で利用可能な状態にします。
    */
    Result Open() NN_NOEXCEPT;

    /**
        @brief ストリームを閉じます。
        @details
            ImportJsonByRapidJson() で利用中は閉じないでください。
     */
    void Close() NN_NOEXCEPT;

    /**
        @brief ストリーミング中にエラーが生じた場合、そのエラー Result が格納されます。
     */
    Result GetResult() const NN_NOEXCEPT;
    int32_t GetHttpCode() NN_NOEXCEPT;

    size_t Receive(void* buffer, size_t bufferSize) NN_NOEXCEPT;
};

class CurlInputStream
    : public CurlInputStreamBase
    , public json::BufferedStreamForRapidJson<CurlInputStream>
{
    NN_DISALLOW_COPY(CurlInputStream);
    // FillBufferImpl を参照可能にするために friend にする必要がある。
    friend class json::BufferedStreamForRapidJson<CurlInputStream>;
private:
    typedef BufferedStreamForRapidJson<CurlInputStream> Base;
    typedef CurlInputStreamBase StreamBase;
    size_t FillBufferImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        return Receive(buffer, bufferSize);
    }

public:
    typedef Base::Ch Ch;
    NN_STATIC_ASSERT(sizeof(Ch) == 1);

public:
    CurlInputStream(CURL* curlHandle, const util::Cancelable* pCancelable = nullptr) NN_NOEXCEPT;

    Result Open() NN_NOEXCEPT
    {
        return StreamBase::Open();
    }

    void Close() NN_NOEXCEPT
    {
        StreamBase::Close();
    }

    Result GetResult() const NN_NOEXCEPT
    {
        return StreamBase::GetResult();
    }

    int32_t GetHttpCode() NN_NOEXCEPT
    {
        return StreamBase::GetHttpCode();
    }

};

}}} // ~namespace nn::http::stream
