﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/http/json/http_RapidJsonConfig.h>
#include <nn/http/http_Result.h>

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#include <rapidjson/writer.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

#include <atomic>
#include <limits>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nn_SdkLog.h>

namespace nn { namespace http { namespace detail {
class AbstractLocalStorage;
}}} // ~namespace nn::http::detail

namespace nn { namespace http { namespace json {

class FixedSizeOneTimeAllocatorForRapidJson
{
    NN_DISALLOW_COPY(FixedSizeOneTimeAllocatorForRapidJson);

public:
    static const bool kNeedFree = false;

private:
    void* m_Buffer;
    size_t m_BufferSize;
    bool m_Allocated;

public:
    FixedSizeOneTimeAllocatorForRapidJson() NN_NOEXCEPT
    {
        NN_ABORT(
            "[nn::http] -----------------------------------------------\n"
            "  ABORT: Default constructor unsupported. (internal)\n");
    }
    FixedSizeOneTimeAllocatorForRapidJson(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(buffer)
        , m_BufferSize(bufferSize)
        , m_Allocated(false)
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(bufferSize > 0);
    }

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(
            !m_Allocated,
            "[nn::http] -----------------------------------------------\n"
            "  ABORT: Can allocate only once. (internal)\n");
        NN_ABORT_UNLESS(
            size <= m_BufferSize,
            "[nn::http] -----------------------------------------------\n"
            "  ABORT: Run out of allocatable buffer: %zu for %zu (internal)\n",
            size, m_BufferSize);
        m_Allocated = true;
        return m_Buffer;
    }

    void* Realloc(void* originalPtr, size_t originalSize, size_t newSize) NN_NOEXCEPT
    {
        NN_UNUSED(originalSize);
        if (originalPtr != nullptr)
        {
            NN_ABORT_UNLESS(originalPtr == m_Buffer);
            m_Allocated = false;
        }
        return Malloc(newSize);
    }

    static void Free(void*) NN_NOEXCEPT {} // default
};

class StringOutputBufferForRapidJson
{
    NN_DISALLOW_COPY(StringOutputBufferForRapidJson);

public:
    typedef char Ch;

private:
    Ch* const m_Buffer;
    const size_t m_BufferSize;
    size_t m_FilledSize;

    Result m_IoResult;

public:
    StringOutputBufferForRapidJson(Ch* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(buffer)
        , m_BufferSize(bufferSize)
        , m_FilledSize(0)
        , m_IoResult(ResultSuccess())
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(bufferSize > 0);
        NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
    }
    ~StringOutputBufferForRapidJson() NN_NOEXCEPT {} // default

    Result Finalize(size_t* pOutSize) NN_NOEXCEPT
    {
        if (m_FilledSize < m_BufferSize)
        {
            m_Buffer[m_FilledSize] = '\0';
        }
        *pOutSize = m_FilledSize;
        NN_RESULT_THROW(m_IoResult);
    }

    void Put(Ch c) NN_NOEXCEPT
    {
        if (!(m_FilledSize < m_BufferSize))
        {
            m_IoResult = ResultInsufficientBuffer();
            return;
        }
        m_Buffer[m_FilledSize ++] = c;
    }
    void Flush() NN_NOEXCEPT  {} // default
};

typedef nne::rapidjson::Writer<StringOutputBufferForRapidJson, nne::rapidjson::UTF8<>, nne::rapidjson::UTF8<>, FixedSizeOneTimeAllocatorForRapidJson>
    DefaultRapidJsonMemoryWriter;

}}} // ~namespace nn::http::json
