﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief   GPIO ドライバ実装提供者向けのインタフェースクラスの宣言
*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/ddsf/ddsf_IDriver.h>
#include <nn/ddsf/ddsf_Result.h>
#include <nn/gpio/gpio_Type.h>
#include <nn/gpio/driver/gpio_PadAccessorDev.h> // for WakePinDebugMode

//---------------------------------------------------------------------------

namespace nn {
namespace gpio {
namespace driver {

    class Pad;

    //! @name GPIO ドライバ実装提供者向け API
    //! @{

    /**
    * @brief   GPIO ドライバ実装が備える必要のあるインタフェースクラスです。
    *
    * @detail
    *   GPIO ドライバ実装提供者は、本クラスを public 継承したオブジェクトを実装し、 nn::gpio::driver::RegisterDriver() で GPIO サブシステムに登録してください。 @n
    *   @n
    *   本クラスから派生クラスへのダウンキャストには、基底クラス @ref nn::ddsf::IDriver の提供する SafeCastTo<>(), SafeCastToPointer<>() メソッドを使用することを強く推奨します。 @n
    *   安全なダウンキャストを使用するには、派生クラスで @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用した型情報タグの定義が必要です。 @n
    *   詳細は @ref nn::ddsf::ICastSafe のリファレンスを参照してください。
    */
    class IGpioDriver :
        public nn::ddsf::IDriver
    {
        NN_DDSF_CAST_SAFE_DECL;

        NN_DISALLOW_COPY(IGpioDriver);
        NN_DISALLOW_MOVE(IGpioDriver);

    public:

        //! @name 生成と破棄
        //! @{

        /**
        * @brief   コンストラクタです。
        */
        IGpioDriver() NN_NOEXCEPT {}

        /**
        * @brief   デストラクタです。
        */
        virtual ~IGpioDriver() NN_NOEXCEPT {}

        //! @}

        //! @name GPIO ドライバ実装の初期化・終了
        //! @{

        /**
        * @brief   ドライバを初期化します。
        *
        * @detail
        *   ドライバを初期化します。 @n
        *   このメソッドは nn::gpio::driver::Initialize() の内部で呼ばれます。 @n
        *   @n
        *   このメソッドは、派生クラスでの実装が必須です。
        */
        virtual void InitializeDriver() NN_NOEXCEPT = 0;

        /**
        * @brief   ドライバを終了します。
        *
        * @detail
        *   ドライバを終了します。 @n
        *   このメソッドは nn::gpio::driver::Finalize() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もしません。
        */
        virtual void FinalizeDriver() NN_NOEXCEPT {}

        //! @}

        //! @name GPIO パッドの初期化・終了
        //! @{

        /**
        * @brief   GPIO パッドを初期化します。
        *
        * @param[in,out]    pPad   GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドを初期化します。 @n
        *   このメソッドは nn::gpio::driver::OpenSession() の内部で、最初にパッドに対するセッションが開かれたときに呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result InitializePad(Pad* pPad) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            return nn::ResultSuccess();
        }

        /**
        * @brief   GPIO パッドをファイナライズします。
        *
        * @param[in,out]    pPad   GPIO パッド
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドをファイナライズします。 @n
        *   このメソッドは nn::gpio::driver::CloseSession() の内部で、パッドに対するオープン状態のセッションが一つもなくなったときに呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もしません。
        */
        virtual void FinalizePad(Pad* pPad) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
        }

        //! @}

        //! @name GPIO パッドの制御
        //! @{

        /**
        * @brief   GPIO パッドの入出力方向を取得します。
        *
        * @param[out]   pOut    入出力方向
        * @param[in]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOut != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの入出力方向を取得します。 @n
        *   このメソッドは nn::gpio::driver::GetDirection(GpioPadSession*) の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetDirection(Direction* pOut, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの入出力方向を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        direction   入出力方向
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの入出力方向を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetDirection() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetDirection(Pad* pPad, Direction direction) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(direction);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの状態 (H/L) を取得します。
        *
        * @param[out]   pOut    GPIO パッドの状態 (H/L)
        * @param[in]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOut != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの状態 (H/L) を取得します。 @n
        *   このメソッドは nn::gpio::driver::GetValue() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetValue(GpioValue* pOut, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの入出力方向を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        value       GPIO パッドの状態 (H/L)
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの状態 (H/L) を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetValue() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetValue(Pad* pPad, GpioValue value) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(value);
            return nn::ddsf::ResultNotSupported();
        }

        //! @}

        //! @name 割り込みハンドリング
        //! @{

        /**
        * @brief   GPIO パッドの割り込みモードを取得します。
        *
        * @param[out]   pOut    GPIO パッドの割り込みモード
        * @param[in]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOut != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの割り込みモードを取得します。 @n
        *   このメソッドは nn::gpio::driver::GetInterruptMode() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetInterruptMode(InterruptMode* pOut, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの割り込み許可状態を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        mode        GPIO パッドの割り込み許可状態
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの割り込みモードを設定します。 @n
        *   このメソッドは nn::gpio::driver::SetInterruptMode() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetInterruptMode(Pad* pPad, InterruptMode mode) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(mode);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの割り込みの有効・無効を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        enable      割り込みの有効・無効
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの割り込みの有効・無効を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetInterruptEnable() または nn::gpio::driver::BindInterrupt() の内部で、 @n
        *   割り込みが有効かつ通知イベントがバインドされている状態になった時に呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetInterruptEnabled(nn::gpio::driver::Pad* pPad, bool enable) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(enable);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの割り込み検出ステータスを取得します。
        *
        * @param[out]   pOut    GPIO パッドの割り込み検出ステータス
        * @param[in]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOut != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの割り込み検出ステータスを取得します。 @n
        *   このメソッドは nn::gpio::driver::GetInterruptStatus() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetInterruptStatus(InterruptStatus* pOut, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの割り込み検出ステータスをクリアします。
        *
        * @param[in,out]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドの割り込み検出ステータスを取得します。 @n
        *   このメソッドは nn::gpio::driver::ClearInterruptStatus() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result ClearInterruptStatus(Pad* pPad) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドの割り込み設定の操作をする際に使用するミューテックスオブジェクトを取得します。
        *
        * @param[in]    pad     GPIO パッド
        *
        * @return
        *   GPIO パッドの割り込み設定の操作をする際に使用するミューテックスオブジェクトへの参照
        *
        * @detail
        *   GPIO パッドの割り込み設定の操作をする際に使用するミューテックスオブジェクトを取得します。 @n
        *   以下のクライアント API 呼び出し時に GPIO サブシステム側がこのメソッドを使用してロックを取得します。 @n
        *
        *   - nn::gpio::driver::BindInterrupt()
        *       - 内部で IGpioDriver::SetInterruptEnabled() を呼ぶことがあります。
        *   - nn::gpio::driver::UnbindInterrupt()
        *       - 内部で IGpioDriver::SetInterruptEnabled() を呼ぶことがあります。
        *   - nn::gpio::driver::GetInterruptStatus()
        *       - 内部で IGpioDriver::GetInterruptStatus() を呼びます。
        *   - nn::gpio::driver::ClearInterruptStatus()
        *       - 内部で IGpioDriver::ClearInterruptStatus() を呼びます。
        *   - nn::gpio::driver::SetInterruptEnable()
        *       - 内部で IGpioDriver::SetInterruptEnabled() を呼ぶことがあります。
        *
        *   ドライバ実装側は、競合するレジスタアクセスなどに対して自身で同じミューテックスを使用して適切な排他制御を行う必要があります。 @n
        *   またドライバ実装側で GPIO パッドに対して以下のメソッドを使用する際も、事前に対応するロックを取得しておく必要があります。 @n
        *
        *   - nn::gpio::driver::Pad::IsAnySessionBoundToInterrupt()
        *   - nn::gpio::driver::Pad::SignalInterruptBoundEvent()
        *
        *   派生クラスでオーバーライドしない場合、このメソッドは既定のグローバルなミューテックスオブジェクトを返します。
        */
        virtual nn::os::SdkMutex& GetInterruptControlMutex(const Pad& pad) const NN_NOEXCEPT
        {
            NN_UNUSED(pad);
            static nn::os::SdkMutex s_DefaultGlobalInterruptControlMutex;
            return s_DefaultGlobalInterruptControlMutex;
        }

        //! @}

        //! @name デバウンスの制御
        //! @{

        /**
        * @brief   チャタリング防止機能の設定状態を取得します。
        *
        * @param[out]   pOut    チャタリング防止機能の設定状態
        * @param[in]    pPad    GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOut != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   チャタリング防止機能の設定状態を取得します。 @n
        *   このメソッドは nn::gpio::driver::GetDebounceEnabled() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetDebounceEnabled(bool* pOut, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   チャタリング防止機能の有効・無効を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        enable      チャタリング防止機能の有効・無効
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   チャタリング防止機能の有効・無効を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetDebounceEnabled() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetDebounceEnabled(Pad* pPad, bool enable) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(enable);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   チャタリング防止機能のスレッショルド時間を取得します。
        *
        * @param[out]   pOutMsec    チャタリング防止機能のスレッショルド時間 [msec]
        * @param[in]    pPad        GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOutMsec != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   チャタリング防止機能のスレッショルド時間を取得します。 @n
        *   このメソッドは nn::gpio::driver::GetDebounceTime() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result GetDebounceTime(int* pOutMsec, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOutMsec);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   チャタリング防止機能のスレッショルド時間を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        msecTime    チャタリング防止機能のスレッショルド時間
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   チャタリング防止機能のスレッショルド時間を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetDebounceTime() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetDebounceTime(Pad* pPad, int msecTime) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(msecTime);
            return nn::ddsf::ResultNotSupported();
        }

        //! @}

        //! @name スリープ関連の制御
        //! @{

        /**
        * @brief   スリープ中の GPIO パッドの状態 (H/L) を設定します。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        value       スリープ中の GPIO パッドの状態 (H/L)
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   スリープ中の GPIO パッドの状態 (H/L) を設定します。 @n
        *   このメソッドは nn::gpio::driver::SetValueForSleepState() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetValueForSleepState(Pad* pPad, GpioValue value) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(value);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   GPIO パッドが起床要因として使われたかどうかを取得します。
        *
        * @param[out]   pOutIsActive    GPIO パッドが起床要因として使われたかどうか
        * @param[in]    pPad            GPIO パッド
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pOutIsActive != nullptr
        *   - pPad != nullptr
        *
        * @detail
        *   GPIO パッドが起床要因として使われたかどうかを取得します。 @n
        *   このメソッドは nn::gpio::driver::IsWakeEventActive() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result IsWakeEventActive(bool* pOutIsActive, Pad* pPad) const NN_NOEXCEPT
        {
            NN_UNUSED(pOutIsActive);
            NN_UNUSED(pPad);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief   次回のスリープからの復帰時以降、指定した GPIO パッドへの IsWakeEventActiveFlag() が強制的に true を返すようにします。
        *
        * @param[in,out]    pPad        GPIO パッド
        * @param[in]        enable      true を指定すると、指定した GPIO パッドへの IsWakeEventActiveFlag() が常に true を返すようになります。 @n
        *                               false を指定すると、この設定を解除します。
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @pre
        *   - pPad != nullptr
        *
        * @detail
        *   次回のスリープからの復帰時以降、指定した GPIO パッドへの IsWakeEventActiveFlag() が強制的に指定した値を返すようにします。 @n
        *   このメソッドは nn::gpio::driver::SetWakeEventActiveFlagSetForDebug() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetWakeEventActiveFlagSetForDebug(Pad* pPad, bool enable) NN_NOEXCEPT
        {
            NN_UNUSED(pPad);
            NN_UNUSED(enable);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief    wake pin のデバッグモードを設定します。
        *
        * @param[in]        mode      wake pin のデバッグモード
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   wake pin のデバッグモードを設定します。 @n
        *   このメソッドは nn::gpio::driver::SetWakePinDebugMode() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ddsf::ResultNotSupported() を返します。
        */
        virtual nn::Result SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT
        {
            NN_UNUSED(mode);
            return nn::ddsf::ResultNotSupported();
        }

        /**
        * @brief    ドライバのスリープ準備を行います。
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   ドライバのスリープ準備を行います。 @n
        *   このメソッドは nn::gpio::driver::Suspend() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result Suspend() NN_NOEXCEPT
        {
            return nn::ResultSuccess();
        }

        /**
        * @brief    ドライバのスリープ準備を行います。こちらの処理はハードウェアがスリープする直前のステート中に呼ばれます。
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   ドライバのスリープ準備を行います。こちらの処理はハードウェアがスリープする直前のステート中に呼ばれます。 @n
        *   このメソッドは nn::gpio::driver::SuspendLow() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result SuspendLow() NN_NOEXCEPT
        {
            return nn::ResultSuccess();
        }

        /**
        * @brief    ドライバのスリープからの復帰準備を行います。
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   ドライバのスリープからの復帰準備を行います。 @n
        *   このメソッドは nn::gpio::driver::Resume() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result Resume() NN_NOEXCEPT
        {
            return nn::ResultSuccess();
        }

        /**
        * @brief    ドライバのスリープからの復帰準備を行います。こちらの処理はハードウェアがスリープから復帰した直後のステート中に呼ばれます。
        *
        * @return
        *   処理に成功した場合 nn::ResultSuccess() を、失敗した場合はそれ以外のリザルトコードを返してください。
        *
        * @detail
        *   ドライバのスリープからの復帰準備を行います。こちらの処理はハードウェアがスリープする直前のステート中に呼ばれます。 @n
        *   このメソッドは nn::gpio::driver::ResumeLow() の内部で呼ばれます。 @n
        *   @n
        *   派生クラスでオーバーライドしない場合、このメソッドは何もせず nn::ResultSuccess() を返します。
        */
        virtual nn::Result ResumeLow() NN_NOEXCEPT
        {
            return nn::ResultSuccess();
        }

        //! @}
    };

    //! @}

} // driver
} // gpio
} // nn
