﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/friends/friends_Types.h>
#include <nn/friends/friends_TypesFriendRequest.h>

namespace nn { namespace friends {

/*!
    @brief      対面フレンド申請の最大件数です。
*/
const int FacedFriendRequestCountMax = 50;

/*!
    @brief      送信可能なフレンド申請の最大件数です。
*/
const int SendFriendRequestCountMax = FriendCountMax;

/*!
    @brief      いっしょに遊んだユーザーの最大件数です。
*/
const int PlayHistoryCountMax = 300;

/*!
    @brief      ユーザーごとに記録するプレイログの最大件数です。
*/
const int PlayLogCountMax = 20;

/*!
    @brief      外部アプリケーション名の最大文字数です。
*/
const int ExternalApplicationNameLengthMax = 256;

/*!
    @brief      Mii の名前の最大文字数です。
*/
const int MiiNameLengthMax = 10;

/*!
    @brief      プレゼンスの公開範囲を表します。
*/
enum PresencePermission
{
    PresencePermission_Self            = 0, //!< 非公開。
    PresencePermission_FavoriteFriends = 1, //!< お気に入りのフレンドまで。
    PresencePermission_Friends         = 2  //!< フレンド全員。
};

/*!
    @brief      拡張プロフィールの公開範囲を表します。
*/
enum ExtraPermission
{
    ExtraPermission_Unconfigured    = 0, //!< 未設定。
    ExtraPermission_Self            = 1, //!< 非公開。
    ExtraPermission_FavoriteFriends = 2, //!< お気に入りのフレンドまで。
    ExtraPermission_Friends         = 3, //!< フレンドまで。
    ExtraPermission_FriendsOfFriend = 4, //!< フレンドのフレンドまで。
    ExtraPermission_Everyone        = 5  //!< 全体公開。
};

/*!
    @brief      フレンド申請リストの種類を表します。
*/
enum RequestListType
{
    RequestListType_Faced    = 0, //!< 対面フレンド申請リスト。
    RequestListType_Sent     = 1, //!< 送信側のフレンド申請リスト。
    RequestListType_Received = 2  //!< 受信側のフレンド申請リスト。
};

/*!
    @brief      フレンド申請手段を表します。
*/
enum RequestType
{
    RequestType_Unknown         = 0, //!< 不明。
    RequestType_Faced           = 1, //!< 対面フレンド申請。
    RequestType_FriendCode      = 2, //!< フレンドコードによるフレンド申請。
    RequestType_InApp           = 3, //!< アプリケーション内、または、いっしょに遊んだ記録からフレンド申請。
    RequestType_FriendCandidate = 4, //!< フレンド候補に対するフレンド申請。
    RequestType_3Ds             = 5, //!< 3DS フレンドに対するフレンド申請。
    RequestType_Nnid            = 6, //!< NNID フレンドに対するフレンド申請。
    RequestType_Facebook        = 7, //!< Facebook フレンドに対するフレンド申請。
    RequestType_Twitter         = 8  //!< Twitter 相互フォロワーに対するフレンド申請。
};

/*!
    @brief      フレンド申請の状態を表します。
*/
enum RequestStatus
{
    RequestStatus_Unknown                                   = 0,  //!< 不明。
    RequestStatus_Pending                                   = 1,  //!< フレンド申請中。
    RequestStatus_Canceled                                  = 2,  //!< 取り消し済み。
    RequestStatus_Authorized                                = 3,  //!< 承認済み。
    RequestStatus_Rejected                                  = 4,  //!< 却下済み。
    RequestStatus_Expired                                   = 5,  //!< 有効期限が切れた。
    RequestStatus_Unsent                                    = 10, //!< 未送信。
    RequestStatus_UnsentForNetworkFailureOccurred           = 11, //!< 未送信（ネットワーク障害のため）。
    RequestStatus_UnsentForMyFriendListLimitReached         = 12, //!< 未送信（自分のフレンドリストが上限に達しているため）。
    RequestStatus_UnsentForFriendListLimitReached           = 13, //!< 未送信（相手のフレンドリストが上限に達しているため）。
    RequestStatus_UnsentForFriendRequestReceptionNotAllowed = 14, //!< 未送信（相手がフレンド申請の受信を許可していないため）。
    RequestStatus_UnsentForBlockedByMe                      = 15  //!< 未送信（自分が相手をブロックリストに登録していたため）。
};

/*!
    @brief      フレンド候補の種類を表します。
*/
enum FriendCandidateType
{
    FriendCandidateType_Undefined           = 0, //!< 未定義。
    FriendCandidateType_ExternalApplication = 1  //!< 外部アプリケーションのフレンド候補。
};

/*!
    @brief      NNID に紐付くアカウントの種類を表します。
*/
enum NintendoNetworkIdAccountType
{
    NintendoNetworkIdAccountType_Undefined = 0, //!< 未定義。
    NintendoNetworkIdAccountType_3Ds       = 1, //!< NNID に紐付く 3DS アカウント。
    NintendoNetworkIdAccountType_Nnid      = 2  //!< NNID 自体のアカウント。
};

/*!
    @brief      SNS アカウントの種類を表します。
*/
enum SnsAccountType
{
    SnsAccountType_Undefined = 0, //!< 未定義。
    SnsAccountType_Facebook  = 1, //!< Facebook。
    SnsAccountType_Twitter   = 2  //!< Twitter。
};

/*!
    @brief      ブロック理由を表します。
*/
enum BlockReason
{
    BlockReason_Unknown          = 0, //!< 不明。
    BlockReason_BadFriendRequest = 1, //!< フレンド申請をブロック。
    BlockReason_BadFriend        = 2, //!< フレンドをブロック。
    BlockReason_InApp            = 3  //!< アプリケーション内、または、いっしょに遊んだ記録からブロック。
};

/*!
    @brief      対面フレンド申請の送信状況を表します。
*/
enum SendingStatus
{
    SendingStatus_Unknown                          = 0, //!< 不明。
    SendingStatus_Unsent                           = 1, //!< 未送信。
    SendingStatus_NetworkFailureOccurred           = 2, //!< ネットワーク障害が発生している。
    SendingStatus_MyFriendListLimitReached         = 3, //!< 自分のフレンドリストが上限に達している。
    SendingStatus_FriendListLimitReached           = 4, //!< 相手のフレンドリストが上限に達している。
    SendingStatus_FriendRequestReceptionNotAllowed = 5, //!< 相手がフレンド申請の受信を許可していない。
    SendingStatus_BlockedByMe                      = 6  //!< 自分が相手をブロックリストに登録していた。
};

/*!
    @brief      フレンド申請 ID です。
*/
struct RequestId
{
    uint64_t value;
};

/*!
    @brief      ユーザー間の関係を表します。
*/
struct Relationship
{
    bool isFriend;         //!< フレンド関係。
    bool isFriendOfFriend; //!< フレンドのフレンド。
    bool isBlocked;        //!< ブロック済み。
    bool isRequestPending; //!< 相手にフレンド申請中。
    NN_PADDING4;
};

/*!
    @brief      経路情報です。

    @details
                フレンドの成立経路やブロックの登録経路等を記録する際のアプリケーション内の情報が取得できます。
*/
struct RouteInfo
{
    ApplicationInfo appInfo;  //!< アプリケーション情報。
    InAppScreenName name;     //!< 相手のアプリ内表示名。
    nn::time::PosixTime time; //!< 登録時刻。
};

/*!
    @brief      プレイログです。

    @details
                ユーザーのアプリケーションのプレイ状況が記録されます。
*/
struct PlayLog
{
    ApplicationInfo appInfo;             //!< アプリケーション情報。
    uint32_t totalPlayCount;             //!< 総プレイ回数。
    uint32_t totalPlayTime;              //!< 総プレイ時間（分）。
    nn::time::PosixTime firstPlayedTime; //!< 最初に遊んだ時間。
    nn::time::PosixTime lastPlayedTime;  //!< 最後に遊んだ時間。
};

/*!
    @brief      いっしょに遊んだ記録です。

    @details
                いっしょに遊んだユーザーやフレンドといっしょに遊んだ記録等の情報が取得できます。
*/
struct PlayRecord
{
    ApplicationInfo appInfo;  //!< アプリケーション情報。
    InAppScreenName name;     //!< 相手のアプリ内表示名。
    InAppScreenName myName;   //!< 自分のアプリ内表示名。
    nn::time::PosixTime time; //!< 登録時刻。
};

/*!
    @brief      いっしょに遊んだユーザーの統計情報です。
*/
struct PlayHistoryStatistics
{
    uint64_t totalLocalPlayCount;  //!< いっしょに遊んだユーザーの登録回数。（ローカルプレイ）
    uint64_t totalOnlinePlayCount; //!< いっしょに遊んだユーザーの登録回数。（オンラインプレイ）
};

/*!
    @brief      フレンドコードです。

    @details
                フレンドコードは、以下のフォーマットで記述されています。

                - ［接頭辞］－［数字 4 桁］－［数字 4 桁］－［数字 4 桁］
*/
struct FriendCode
{
    char value[32]; //!< フレンドコード文字列。
};

/*!
    @brief      外部アプリケーションのカタログ ID です。

    @details
                ターゲットプラットフォーム外のアプリケーション情報を取得する時に使用します。
*/
struct ExternalApplicationCatalogId
{
    Bit64 value[2];
};

/*!
    @brief      外部アプリケーション名です。

    @details
                ターゲットプラットフォーム外のアプリケーション情報を取得する時に使用します。

                外部アプリケーション名は UTF-8 でエンコードされており、最大文字数は @ref ExternalApplicationNameLengthMax です。
*/
struct ExternalApplicationName
{
    char name[1024];                     //!< 外部アプリケーション名を格納するバッファ。
    nn::settings::LanguageCode language; //!< 言語コード。
};

/*!
    @brief      外部アプリケーションのカタログです。

    @details
                ターゲットプラットフォーム外のアプリケーション情報を取得する時に使用します。
*/
struct ExternalApplicationCatalog
{
    ExternalApplicationCatalogId id; //!< カタログ ID。
    ExternalApplicationName name;    //!< アプリケーション名。
    Url iconImageUrl;                //!< アプリケーションアイコンの URL。
};

/*!
    @brief      Mii の名前です。

    @details
                名前は UTF-8 でエンコードされています。
*/
struct MiiName
{
    char value[32]; //!< 名前を格納するバッファ。
};

/*!
    @brief      Mii 画像 URL を構成するためのパラメータです。
*/
struct MiiImageUrlParam
{
    char value[16]; //!< 値。
};

/*!
    @brief      NNID のユーザー情報です。
*/
struct NintendoNetworkIdUserInfo
{
    MiiName miiName;                   //!< Mii の名前。
    MiiImageUrlParam miiImageUrlParam; //!< Mii 画像 URL パラメータ。
    int32_t friendCount;               //!< フレンド人数。
    NN_PADDING4;
};

/*!
    @brief      SNS アカウントの紐付け状態を表します。
*/
struct SnsAccountLinkage
{
    bool isFacebookLinked; //!< Facebook と紐付いているかどうか。
    bool isTwitterLinked;  //!< Twitter と紐付いているかどうか。
    NN_PADDING6;
};

/*!
    @brief      SNS アカウントのプロフィール名です。

    @details
                プロフィール名は UTF-8 でエンコードされています。
*/
struct SnsAccountProfileName
{
    char value[256]; //!< プロフィール名を格納するバッファ。
};

/*!
    @brief      SNS アカウントのプロフィール画像の URL です。
*/
struct SnsAccountProfileImageUrl
{
    char value[384]; //!< URL を格納するバッファ。
};

/*!
    @brief      SNS アカウントのプロフィールです。

    @details
                SNS によってはユーザーを識別するための名前を複数持っている場合があります。

                システムは最大 2 つまで名前を取得します。@n
                名前が 1 つしかない場合、プロフィール名 2 は空文字列になります。
*/
struct SnsAccountProfile
{
    SnsAccountProfileName name1;        //!< プロフィール名 1。
    SnsAccountProfileName name2;        //!< プロフィール名 2。
    SnsAccountProfileImageUrl imageUrl; //!< プロフィール画像 URL。
};

/*!
    @brief      WEB ページ URL です。

    @details
                WEB ページの URL を取得するために利用します。@n
                文字列は NULL 終端されます。

                クエリ文字列を含む非常に長い文字列になる可能性があります。
*/
struct WebPageUrl
{
    char value[4096]; //!< URL を格納するバッファ。
};

}}
