﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/json/friends_Json.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

/*!
    @brief      JSON パッチリクエストを生成するクラスです。

    @details
                JSON パッチに関しては、RFC 6902 を参照してください。
*/
class JsonPatchGenerator
{
public:
    /*!
        @brief      パッチの書き込みを開始します。

        @return     処理結果。

        @details
                    本関数は、最初に 1 度だけ呼び出してください。
    */
    static bool BeginWrite(JsonWriter& writer) NN_NOEXCEPT;

    /*!
        @brief      パッチの書き込みを終了します。

        @return     処理結果。

        @details
                    本関数は、最後に 1 度だけ呼び出してください。
    */
    static bool EndWrite(JsonWriter& writer) NN_NOEXCEPT;

    /*!
        @brief      add パッチを書き込みを開始します。

        @param[in]  path    パス。

        @pre
            - path != nullptr

        @return     処理結果。

        @details
                    本関数を呼び出し後、バリューを書き込んでください。@n
                    書き込み後、 @ref EndWriteAppOperation を呼び出してください。
    */
    static bool BeginWriteAddOperation(JsonWriter& writer, const char* path) NN_NOEXCEPT;

    /*!
        @brief      add パッチを書き込みを終了します。

        @return     処理結果。
    */
    static bool EndWriteAddOperation(JsonWriter& writer) NN_NOEXCEPT;

    /*!
        @brief      replace パッチを書き込みを開始します。

        @param[in]  path    パス。

        @pre
            - path != nullptr

        @return     処理結果。

        @details
                    本関数を呼び出し後、バリューを書き込んでください。@n
                    書き込み後、 @ref EndWriteReplaceOperation を呼び出してください。
    */
    static bool BeginWriteReplaceOperation(JsonWriter& writer, const char* path) NN_NOEXCEPT;

    /*!
        @brief      replace パッチを書き込みを終了します。

        @return     処理結果。
    */
    static bool EndWriteReplaceOperation(JsonWriter& writer) NN_NOEXCEPT;

    /*!
        @brief      move パッチを書き込みます。

        @param[in]  from    参照元パス。
        @param[in]  path    参照先パス。

        @pre
            - from != nullptr
            - path != nullptr

        @return     処理結果。
    */
    static bool WriteMoveOperation(JsonWriter& writer, const char* from, const char* path) NN_NOEXCEPT;

    /*!
        @brief      copy パッチを書き込みます。

        @param[in]  from    参照元パス。
        @param[in]  path    参照先パス。

        @pre
            - from != nullptr
            - path != nullptr

        @return     処理結果。
    */
    static bool WriteCopyOperation(JsonWriter& writer, const char* from, const char* path) NN_NOEXCEPT;

    /*!
        @brief      remove パッチを書き込みます。

        @param[in]  path    パス。

        @pre
            - path != nullptr

        @return     処理結果。
    */
    static bool WriteRemoveOperation(JsonWriter& writer, const char* path) NN_NOEXCEPT;
};

}}}}}
