﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      対面フレンド申請リストの管理モジュールです。
*/
class FacedFriendRequestManager
{
private:
    NN_DISALLOW_COPY(FacedFriendRequestManager);
    NN_DISALLOW_MOVE(FacedFriendRequestManager);

public:
    /*!
        @brief      対面フレンド申請登録キーです。
    */
    struct FacedFriendRequestRegistrationKeyData
    {
        // 1
        Bit8 version;
        // 1
        Bit8 platform;
        // 1
        Bit8 hashKeyIndex;
        // 5
        char reserved1[5];
        // 8
        nn::account::NetworkServiceAccountId accountId;
        // 16
        char reserved2[16];
        // 32
        Bit8 hash[32];
    };

    NN_STATIC_ASSERT(sizeof (FacedFriendRequestRegistrationKeyData) == sizeof (FacedFriendRequestRegistrationKey));

    /*!
        @brief      レコードです。
    */
    struct Record
    {
        FacedFriendRequestResource resource;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 216);

private:
    /*!
        @brief      コンストラクタです。
    */
    FacedFriendRequestManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static FacedFriendRequestManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(FacedFriendRequestManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      対面フレンド申請の登録キーを取得します。

        @param[out] outKey  登録キー。
        @param[in]  uid     ユーザーアカウント。

        @return     処理結果。

        @pre
            - outKey != nullptr
    */
    nn::Result GetFacedFriendRequestRegistrationKey(FacedFriendRequestRegistrationKey* outKey,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請を追加します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  key         登録キー。
        @param[in]  nickname    相手のニックネーム。
        @param[in]  image       相手のプロフィール画像。
        @param[in]  imageSize   相手のプロフィール画像のサイズ。

        @return     処理結果。

        @pre
            - image != nullptr
            - imageSize > 0
    */
    nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
        const FacedFriendRequestRegistrationKey& key, const nn::account::Nickname& nickname,
        const void* image, size_t imageSize) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請を追加します。

        @param[in]  uid                 ユーザーアカウント。
        @param[in]  key                 登録キー。
        @param[in]  nickname            相手のニックネーム。
        @param[in]  image               相手のプロフィール画像。
        @param[in]  imageSize           相手のプロフィール画像のサイズ。
        @param[in]  appInfo             アプリケーション情報。
        @param[in]  inAppScreenName     相手のアプリ内表示名。
        @param[in]  myInAppScreenName   自分のアプリ内表示名。

        @return     処理結果。

        @pre
            - image != nullptr
            - imageSize > 0
    */
    nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
        const FacedFriendRequestRegistrationKey& key, const nn::account::Nickname& nickname,
        const void* image, size_t imageSize,
        const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請を取り下げます。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   相手のネットワークサービスアカウント ID。

        @return     処理結果。
    */
    nn::Result CancelFacedFriendRequest(const nn::account::Uid& uid,
        nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請リストを取得します。

        @param[out] outCount    取得した数。
        @param[out] outRequests フレンド申請リスト。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  offset      オフセット。
        @param[in]  count       フレンド申請リストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outRequests != nullptr
    */
    nn::Result GetFacedFriendRequestList(int* outCount, FriendRequestImpl* outRequests,
        const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請リソースを取得します。

        @param[out] outResource フレンド申請リソース。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   相手のネットワークサービスアカウント ID。

        @return     処理結果。

        @pre
            - outResource != nullptr
    */
    nn::Result GetFacedFriendRequest(FacedFriendRequestResource* outResource,
        const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;

    /*!
        @brief      フレンド申請リソースリストを取得します。

        @param[out] outCount        取得した数。
        @param[out] outResources    フレンド申請リソースリスト。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  offset          オフセット。
        @param[in]  count           フレンド申請リソースリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outResources != nullptr
    */
    nn::Result GetFacedFriendRequestList(int* outCount, FacedFriendRequestResource* outResources,
        const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      プロフィール画像を取得します。

        @param[out] outSize     取得したサイズ。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   相手のネットワークサービスアカウント ID。
        @param[in]  buffer      バッファ。
        @param[in]  size        バッファのサイズ。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - buffer != nullptr
            - size > 0
    */
    nn::Result GetProfileImage(size_t* outSize,
        const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      プロフィール画像を取得します。

        @param[out] outSize     取得したサイズ。
        @param[in]  url         URL。
        @param[in]  buffer      バッファ。
        @param[in]  size        バッファのサイズ。

        @return     処理結果。

        @pre
            - outSize != nullptr
            - url != nullptr
            - buffer != nullptr
            - size > 0
    */
    nn::Result GetProfileImage(size_t* outSize,
        const char* url, void* buffer, size_t size) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請リストが空かどうかを判定します。

        @param[out] outIsEmpty  対面フレンド申請リストが空かどうか。
        @param[in]  uid         ユーザーアカウント。

        @return     処理結果。

        @pre
            - outIsEmpty != nullptr
    */
    nn::Result IsEmpty(bool* outIsEmpty, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      対面フレンド申請リストを更新します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resources   フレンド申請リソースリスト。
        @param[in]  count       フレンド申請リソースリストの要素数。

        @return     処理結果。

        @pre
            - resources != nullptr
            - count <= FacedFriendRequestCountMax

        @details
                    指定したリソースのフレンド申請の状態が（ネットワーク上で）フレンド申請中であった場合、リストから除外されます。
    */
    nn::Result UpdateFacedFriendRequestList(const nn::account::Uid& uid,
        const FacedFriendRequestResource* resources, int count) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Records[FacedFriendRequestCountMax];
    int m_Count;

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    bool Verify(int count) NN_NOEXCEPT;
    //
    nn::Result SaveProfileImage(nn::account::NetworkServiceAccountId accountId,
        const void* image, size_t imageSize) NN_NOEXCEPT;
    nn::Result DeleteProfileImage(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT;
    //
    nn::Result UpdateFacedFriendRequest(const FacedFriendRequestResource& resource) NN_NOEXCEPT;
};

}}}}}
