﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/friends/detail/service/util/friends_Cancelable.h>
#include <nn/account.h>
#include <nn/account/account_ApiBaasAccessToken.h>
#include <nn/account/account_ApiForSystemServices.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      アカウントライブラリのラッパーモジュールです。
*/
class Account
{
public:
    /*!
        @brief      ネットワークサービスアクセストークンです。
    */
    struct NetworkServiceAccessToken
    {
        char value[nn::account::BaasAccessTokenSizeMax + 1];
    };

    /*!
        @brief      ニンテンドーアカウントの ID トークンです。
    */
    struct NintendoAccountIdToken
    {
        char value[nn::account::NintendoAccountIdTokenLengthMax + 1];
    };

public:
    /*!
        @brief      ネットワークサービスアカウント ID を取得します。

        @param[out] outId   ネットワークサービスアカウント ID。
        @param[in]  uid     ユーザーアカウント。

        @return     処理結果。
    */
    static nn::Result GetNetworkServiceAccountId(nn::account::NetworkServiceAccountId* outId,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      デバイスアカウント ID を取得します。

        @param[out] outId   デバイスアカウント ID。
        @param[in]  uid     ユーザーアカウント。

        @return     処理結果。
    */
    static nn::Result GetDeviceAccountId(uint64_t* outId,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアクセストークンの取得を要求します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  cancelable  キャンセル可能オブジェクト。

        @return     処理結果。
    */
    static nn::Result EnsureNetworkServiceAccessToken(const nn::account::Uid& uid,
        const detail::service::util::Cancelable& cancelable) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアクセストークンを取得します。

        @param[out] outToken    ネットワークサービスアクセストークン。
        @param[in]  uid         ユーザーアカウント。

        @return     処理結果。
    */
    static nn::Result GetNetworkServiceAccessToken(NetworkServiceAccessToken* outToken,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントが Open 状態かどうかを取得します。

        @param[in]  uid ユーザーアカウント。

        @return     ユーザーアカウントが Open 状態かどうか。
    */
    static bool IsUserOpened(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ネットワークサービスアカウントが有効かどうかを取得します。

        @param[in]  uid ユーザーアカウント。

        @return     ネットワークサービスアカウントが有効かどうか。
    */
    static bool IsNetworkServiceAccountAvailable(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      子どもアカウントかどうかを判定します。

        @param[in]  uid ユーザーアカウント。

        @return     子どもアカウントかどうか。

        @details
                    ニンテンドーアカウントが紐付いていないユーザーアカウントを指定した場合、子どもアカウントであると判定します。
    */
    static bool IsChild(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      フレンド候補用の ID トークンを取得します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  cancelable  キャンセル可能オブジェクト。

        @return     処理結果。

        @details
                    本関数は、最後に取得した ID トークンをキャッシュします。@n
                    指定したユーザーアカウントに紐付くニンテンドーアカウントの ID トークンキャッシュが存在する場合、それを返します。

                    ID トークンの有効期限が切れていた場合、 @ref ClearNintendoAccountIdTokenForFriendCandidate でキャッシュをクリアした後、処理を再試行してください。
    */
    static nn::Result GetNintendoAccountIdTokenForFriendCandidate(NintendoAccountIdToken* outToken,
        const nn::account::Uid& uid, const detail::service::util::Cancelable& cancelable) NN_NOEXCEPT;

    /*!
        @brief      フレンド候補用の ID トークンキャッシュをクリアします。
    */
    static void ClearNintendoAccountIdTokenForFriendCandidate() NN_NOEXCEPT;
};

}}}}}
