﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>

#if !defined (NN_BUILD_CONFIG_COMPILER_GCC) // gcc4.x では、 is_trivially_copyable が使えない

/*!
    @brief      Impl クラスの生成を開始します。

    @param[in]  cls Impl を生成するクラス。

    @details
                本マクロで作成した Impl のオブジェクトは、元のクラスに memcpy でコピーすることが可能です。@n
                Impl クラスを生成するには、元のクラスが以下の条件にあてはまる必要があります。

                @li standard-layout class であること
                @li trivially-copyable class であること

                本マクロは、NN_DETAIL_FRIENDS_END_IMPL と組み合わせて利用します。

                利用例：
                    NN_DETAIL_FRIENDS_BEGIN_IMPL(Friend)
                    {
                        ...
                    }
                    NN_DETAIL_FRIENDS_END_IMPL(Friend);

                Impl オブジェクトは IPC でやり取りされる前提のため、各メンバーは POD 型にする必要があります。
*/
#define NN_DETAIL_FRIENDS_BEGIN_IMPL(cls)                       \
                                                                \
    NN_STATIC_ASSERT(std::is_standard_layout<cls>::value);      \
    NN_STATIC_ASSERT(std::is_trivially_copyable<cls>::value);   \
                                                                \
    struct cls##Impl_

#else

#define NN_DETAIL_FRIENDS_BEGIN_IMPL(cls)                       \
                                                                \
    NN_STATIC_ASSERT(std::is_standard_layout<cls>::value);      \
                                                                \
    struct cls##Impl_

#endif

/*!
    @brief      Impl クラスの生成を完了します。

    @param[in]  cls Impl を生成するクラス。

    @details
                本マクロは、Impl クラスのサイズを元クラスのバッファサイズに合わせるためにパディング領域を挟み込みます。
*/
#define NN_DETAIL_FRIENDS_END_IMPL(cls)                         \
    ;                                                           \
    NN_STATIC_ASSERT(std::is_pod<cls##Impl_>::value);           \
                                                                \
    struct cls##Impl                                            \
    {                                                           \
        union                                                   \
        {                                                       \
            cls##Impl_ data;                                    \
            nn::Bit8 padding[sizeof (cls)];                     \
        };                                                      \
    };                                                          \
    NN_STATIC_ASSERT(std::is_pod<cls##Impl>::value);            \
                                                                \
    NN_STATIC_ASSERT(sizeof (cls##Impl) == sizeof (cls))

/*!
    @brief      Impl メンバーへのアクセサを取得します。

    @param[in]  cls クラス。

    @return     アクセサ。
*/
#define NN_DETAIL_FRIENDS_GET_IMPL(cls) reinterpret_cast<detail::cls##Impl_*>(&m_Storage)

/*!
    @brief      Impl メンバーへの const アクセサを取得します。

    @param[in]  cls クラス。

    @return     const アクセサ。
*/
#define NN_DETAIL_FRIENDS_CONST_GET_IMPL(cls) reinterpret_cast<const detail::cls##Impl_*>(&m_Storage)

/*!
    @brief      Impl オブジェクトをクリアします。
*/
#define NN_DETAIL_FRIENDS_CLEAR_IMPL() \
    std::memset(&m_Storage, 0, sizeof (m_Storage));
