﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ファン制御ライブラリ実装部の制御機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/fan/fan_ControlTypes.h>

namespace nn {
namespace fan {
namespace impl {

namespace detail {
    const int ControllerSize = 64;
    const int ControllerAlignment = NN_ALIGNOF(void*);
    struct ControllerImplPadded;
}

/**
 * @brief コントローラ用パラメータです。
 *
 * @details コントローラ用パラメータです。内部の変数に直接アクセスしないでください。
 */
struct Controller
{
    nn::util::TypedStorage<detail::ControllerImplPadded, detail::ControllerSize, detail::ControllerAlignment> _impl;
};

//! @name コントローラの初期化と終了
//! @{

/**
 * @brief 指定したファンのコントローラをオープンします。
 *
 * @param[out]  pOutController オープンされたコントローラ
 * @param[in]   name           ファン識別子
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  指定したファンは存在している必要があります。@n
 *  @a pOutController はアクセス可能なアドレスを指している必要があります。
 *
 * @post
 *  指定したファンが初期状態でアクセス可能な状態になります。@n
 *  また、@a pOutController に有効なコントローラが格納されます。
 *
 * @details
 *  指定したファンのコントローラをオープンし、ファンの制御を可能にします。
 */
void OpenController(Controller* pOutController, FanName name) NN_NOEXCEPT;

/**
 * @brief 指定したファンのコントローラをクローズします。
 *
 * @param[in]   pController     コントローラ
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。
 *
 * @post
 *  @a pController が無効なコントローラになります。
 *
 * @details
 *  指定したファンのコントローラをクローズします。
 */
void CloseController(Controller* pController) NN_NOEXCEPT;

//! @}
// ~コントローラの初期化と終了

//! @name 回転速度の制御
//! @{

/**
 * @brief ファンの目標回転速度レベルを設定します。
 *
 * @param[in]   pController     コントローラ
 * @param[in]   level           目標回転速度レベル
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *  @a level は 0 以上 @ref RotationSpeedLevelMax 以下である必要があります。@n
 *
 * @post
 *  ファンの回転が @a level に対応した速度に徐々に変化します。
 *
 * @details
 *  ファンの目標回転速度レベルを設定します。@n
 *  @a level が 0 のときファンは停止状態となり、@ref RotationSpeedLevelMax のとき最大速度で回転します。@n
 */
void SetRotationSpeedLevel(Controller* pController, RotationSpeedLevel level) NN_NOEXCEPT;

/**
 * @brief ファンの目標回転速度レベルを現在のレベルからの差分で設定します。
 *
 * @param[in]   pController     コントローラ
 * @param[in]   levelDiff       目標回転速度レベルの差分
 *
 * @return      設定後の目標回転速度レベル
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *
 * @post
 *  ファンの回転が戻り値で返したレベルに対応する速度に徐々に変化します。
 *
 * @details
 *  ファンの目標回転速度レベルを現在のレベルからの差分で設定します。@n
 *  指定された差分の適用によって目標回転速度レベルが 0 を下回る場合は、0 に設定されます。@n
 *  指定された差分の適用によって目標回転速度レベルが @ref RotationSpeedLevelMax を上回る場合は、
 *  @ref RotationSpeedLevelMax に設定されます。@n
 *  戻り値には、適用後の目標回転速度レベルを返します。
 */
RotationSpeedLevel SetRotationSpeedLevelByDiff(Controller* pController, int levelDiff) NN_NOEXCEPT;

/**
 * @brief ファンの現在の目標回転速度レベルを取得します。
 *
 * @param[in]   pController    コントローラ
 *
 * @return      目標回転速度レベル
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *
 * @details
 *  ファンの現在の目標回転速度レベルを取得します。@n
 */
RotationSpeedLevel GetRotationSpeedLevel(Controller* pController) NN_NOEXCEPT;

/**
 * @brief ファンの現在の実際の回転速度レベルを取得します。
 *
 * @param[in]   pController    コントローラ
 *
 * @return      実際の回転速度レベル
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *
 * @details
 *  ファンの現在の実際の回転速度レベルを取得します。@n
 */
RotationSpeedLevel GetActualRotationSpeedLevel(Controller* pController) NN_NOEXCEPT;

//! @}
// ~回転速度の制御

//! @name 回転速度の直接指定による制御（開発用途限定）
//! @{

/**
 * @brief ファンの回転速度レートを設定します。
 *
 * @param[in]   pController     コントローラ
 * @param[in]   rotationRate    回転速度レート
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *  @a rotationRate は [0 - MaxRotationRate] の整数値である必要があります。
 *
 * @post
 *  ファンの回転が @a rotationRate に対応した速度になります。
 *
 * @details
 *  ファンの回転速度レートを設定します。@n
 *  @a rotationRate が 0 のときファンは停止状態となり、 MaxRotationRate のとき最大速度で回転します。@n
 *  本関数は開発目的での利用を想定しており、製品環境では利用できません。@n
 *  製品環境では @ref SetRotationSpeedLevel() や @ref SetRotationSpeedLevelByDiff() を使用してください。@n
 */
void SetRotationRate(Controller* pController, int rotationRate) NN_NOEXCEPT;

/**
 * @brief ファンの現在の回転速度レートを取得します。
 *
 * @param[in]   pController    コントローラ
 *
 * @return      回転速度レート
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *
 * @details
 *  ファンの現在の回転速度レートを取得します。@n
 *  本関数は開発目的での利用を想定しており、製品環境では利用できません。@n
 *  製品環境では @ref GetRotationSpeedLevel() を使用してください。@n
 */
int GetRotationRate(Controller* pController) NN_NOEXCEPT;

//! @}
// ~回転速度の直接指定による制御（開発用途限定）

//! @name ファンの有効化と無効化
//! @{

/**
 * @brief     条件を満たしている場合にファンを回転させます。
 *
 * @param[in] pController コントローラ
 *
 * @pre       ライブラリは初期化済の状態である必要があります。@n
 *
 * @details   条件を満たしている場合にファンを回転させます。@n
 *            @ref SetRotationSpeedLevel で指定された回転速度レベルに向けて徐々に回転数を変更します。@n
 *            @ref SetRotationSpeedLevel で有効な回転速度レベルが設定されていないか、
 *            @ref DisableFanFromServer が呼ばれている場合、ファンは回転しません。
 */
void EnableFan(Controller* pController) NN_NOEXCEPT;

/**
 * @brief     ファンを停止させます。
 *
 * @param[in] pController コントローラ
 *
 * @pre       ライブラリは初期化済の状態である必要があります。@n
 *
 * @details   ファンを停止させます。@n
 *            回転数は急峻に減少します。@n
 *            回転数を徐々に減少させたい場合は @ref SetRotationSpeedLevel を使用してください。
 */
void DisableFan(Controller* pController) NN_NOEXCEPT;

/**
 * @brief   条件を満たしている全てのファンを回転させます。@n
 *          ファンコントローラサーバ用 API です。
 *
 * @pre     ライブラリは初期化済の状態である必要があります。
 *
 * @details 条件を満たしている全てのファンを回転させます。@n
 *          ファンコントローラサーバ用 API です。@n
 *          @ref SetRotationSpeedLevel で指定された回転速度レベルに向けて徐々に回転数を変更します。@n
 *          @ref SetRotationSpeedLevel で有効な回転速度レベルが設定されていないか、
 *          @ref DisableFan が呼ばれているファンは回転しません。
 */
void EnableFanFromServer() NN_NOEXCEPT;

/**
 * @brief   全てのファンを停止させます。@n
 *          ファンコントローラサーバ用 API です。
 *
 * @pre     ライブラリは初期化済の状態である必要があります。
 *
 * @details 全てのファンを停止させます。@n
 *          ファンコントローラサーバ用 API です。@n
 *          回転数は急峻に減少します。
 */
void DisableFanFromServer() NN_NOEXCEPT;

//! @}
// ~ファンの有効化と無効化

} // impl
} // fan
} // nn
