﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  eTicketService の権利判定に関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/es/es_RightsTypes.h>
#include <nn/es/es_Types.h>

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace es {

/**
*   @brief      全てのアカウントで利用可能な権利を持っているかを判定します。
*
*   @param[out] outStatusList                   取得した権利の状態を格納する配列
*   @param[in]  statusCount                     権利の状態の配列の要素数
*   @param[in]  rightsIdList                    権利の状態を確認する 128bit RightsId の配列
*   @param[in]  rightsIdCount                   RightsId の配列の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre
*       - statusCount ==  rightsIdCount
*/
Result CheckRightsStatus(RightsStatus outStatusList[], int statusCount, const nn::es::RightsIdIncludingKeyId rightsIdList[], int rightsIdCount) NN_NOEXCEPT;

/**
*   @brief      指定したアカウントが利用可能な権利を持っているかを判定します。
*               複数のアカウントが指定された場合は、指定された全てのアカウントが利用可能な権利を持っている場合のみ権利を持っていると判定されます。
*
*   @param[out] outStatusList                   取得した権利の状態を格納する配列
*   @param[in]  statusCount                     権利の状態の配列の要素数
*   @param[in]  rightsIdList                    権利の状態を確認する 128bit RightsId の配列
*   @param[in]  rightsIdCount                   RightsId の配列の要素数
*   @param[in]  naIdList                        権利を判定するアカウントの配列
*   @param[in]  naIdCount                       アカウントの配列の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre
*       - statusCount ==  rightsIdCount
*/
Result CheckRightsStatus(RightsStatus outStatusList[], int statusCount, const nn::es::RightsIdIncludingKeyId rightsIdList[], int rightsIdCount, account::NintendoAccountId naIdList[], int naIdCount) NN_NOEXCEPT;

/**
*   @brief      全てのアカウントで利用可能な権利を持っているかを判定します。
*
*   @param[out] outStatusList                   取得した権利の状態を格納する配列
*   @param[in]  statusCount                     権利の状態の配列の要素数
*   @param[in]  rightsIdList                    権利の状態を確認する RightsId の配列
*   @param[in]  rightsIdCount                   RightsId の配列の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre
*       - statusCount ==  rightsIdCount
*/
Result CheckRightsStatus(RightsStatus outStatusList[], int statusCount, const nn::es::RightsId rightsIdList[], int rightsIdCount) NN_NOEXCEPT;

/**
*   @brief      指定したアカウントが利用可能な権利を持っているかを判定します。
*               複数のアカウントが指定された場合は、指定された全てのアカウントが利用可能な権利を持っている場合のみ権利を持っていると判定されます。
*
*   @param[out] outStatusList                   取得した権利の状態を格納する配列
*   @param[in]  statusCount                     権利の状態の配列の要素数
*   @param[in]  rightsIdList                    権利の状態を確認する RightsId の配列
*   @param[in]  rightsIdCount                   RightsId の配列の要素数
*   @param[in]  naIdList                        権利を判定するアカウントの配列
*   @param[in]  naIdCount                       アカウントの配列の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre
*       - statusCount ==  rightsIdCount
*/
Result CheckRightsStatus(RightsStatus outStatusList[], int statusCount, const nn::es::RightsId rightsIdList[], int rightsIdCount, account::NintendoAccountId naIdList[], int naIdCount) NN_NOEXCEPT;

/**
*   @brief      対象 RightsId のタイトル鍵を登録して利用中の権利の情報に追加します。
*
*   @param[in]  rightsIdList                    タイトル鍵のを登録する 128bit RightsId の配列
*   @param[in]  keyGenerationList               タイトル鍵の鍵世代の配列
*   @param[in]  count                           各配列の要素数
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultRightsNotAvailable        権利が利用できません。
*   @retval     ResultTitleKeyAlreadyRegistered タイトル鍵が既に登録されています。
*   @retval     ResultTicketInvalid             チケットが不正です。
*   @retval     ResultCertificateInvalid        証明書が不正です。
*   @retval     ResultRightsIdNotFound          RightsId が見つかりませんでした。
*   @retval     ResultCertificateNotFound       証明書が見つかりませんでした。
*   @retval     ResultTitleKeyCouldntDecrypt    TitleKey の復号に失敗しました。
*
*   @details    内部で登録されているアカウントが権利を持っているかの判定を行い、権利がある場合にしか登録できません。
*/
Result RegisterTitleKey(const nn::es::RightsIdIncludingKeyId rightsIdList[], const int keyGenerationList[], int count) NN_NOEXCEPT;

/**
*   @brief      タイトル鍵の登録を全解除して利用中の権利の情報を全削除します。
*/
void UnregisterAllTitleKey() NN_NOEXCEPT;

/**
*   @brief      アカウントに限定された権利の利用者アカウントを登録します。
*
*   @param[in]  naId                            登録するアカウントの NaId
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultUserListFull              ユーザリストの上限数を超えています。
*   @retval     ResultUserAlreadyRegistered     既にユーザが登録されています。
*/
Result RegisterAccountRestrictedRightsUser(account::NintendoAccountId naId) NN_NOEXCEPT;

/**
*   @brief      アカウントに限定された権利の利用者アカウントの登録を全解除します。
*/
void UnregisterAllAccountRestrictedRightsUser() NN_NOEXCEPT;

/**
*   @brief      アカウントに限定された権利の利用者アカウントのリストを取得します。
*
*   @param[out] outCount                        登録されているアカウントの数
*   @param[out] outNaIdList                     登録されているアカウントの NaId を格納する配列
*   @param[in]  count                           登録されているアカウントの NaId を格納する配列の要素数
*
*   @return     アカウントに限定された権利の利用者が登録されているか
*
*   @pre
*       - count >=  account::UserCountMax
*
*   @details    戻り値が true の場合、アカウントに限定された権利の利用者アカウントが登録されており、outCount と outNaIdList にその情報が格納されます。
*               戻り値が false の場合、outCount と outNaIdList の内容は保証されません。
*/
bool ListAccountRestrictedRightsUser(int* outCount, account::NintendoAccountId outNaIdList[], int count) NN_NOEXCEPT;

}}

