﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 内部システム用エラー表示用APIの定義
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/err/err_SystemErrorArg.h>
#include <nn/err/err_Types.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace err {

//! @name 本体システム向けエラー表示用API
//! @{

/**
* @brief        表示用の Result と障害解析用に保存する Result のバックトレースを指定してエラービューアを表示します。
* @param[in]    displayResult エラービューアに表示する Result
* @param[in]    resultBacktrace 障害解析用に保存する Result のバックトレース
*/
void ShowError(Result displayResult, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT;

/**
* @brief        障害解析用に保存するエラーコンテキストを指定してエラービューアを表示します。
* @param[in]    result エラービューアに表示する Result
* @param[in]    errorContext 障害解析用に保存するエラーコンテキスト。
*/
void ShowError(Result result, const ErrorContext& errorContext) NN_NOEXCEPT;

/**
* @brief        障害解析用に保存するエラーコンテキストを指定してエラービューアを表示します。
* @param[in]    errorCode エラービューアに表示する ErrorCode
* @param[in]    errorContext 障害解析用に保存するエラーコンテキスト。
*/
void ShowError(ErrorCode errorCode, const ErrorContext& errorContext) NN_NOEXCEPT;

/**
* @brief        エラーコードとメッセージを指定してエラービューアを表示します。
* @param[in]    arg エラーの情報を保持するクラス
* @pre
*  - arg のダイアログ表示用のメッセージと全画面表示用のメッセージのどちらか一方、もしくは両方にメッセージが設定されている。
*  - arg の言語が設定されている。
* @details
* - エラー表示を行っている間は呼び出し元をブロックします。
* - 本関数は「エラー表示関数」の1つです。
*/
void ShowSystemError(const SystemErrorArg& arg) NN_NOEXCEPT;

/**
* @brief        エラービューアからの他のアプレット（アプリケーション含む）へのジャンプを無効にした状態でエラービューアを起動します。
* @param[in]    result 発生したエラー
* @return       ジャンプ先
* @details      エラービューアから直接ジャンプが行われると不都合がある場合はこちらを利用ください。
*               戻り値が Nowhere 以外の場合、ExecuteJump() に渡すことでジャンプを実行します。
*/
ErrorViewerJumpDestination ShowErrorWithoutJump(nn::Result result) NN_NOEXCEPT;

/**
* @brief        エラービューアからの他のアプレット（アプリケーション含む）へのジャンプを無効にした状態でエラービューアを起動します。
* @param[in]    errorCode 発生したエラー
* @return       ジャンプ先
* @details      エラービューアから直接ジャンプが行われると不都合がある場合はこちらを利用ください。
*               戻り値が Nowhere 以外の場合、ExecuteJump() に渡すことでジャンプを実行します。
*/
ErrorViewerJumpDestination ShowErrorWithoutJump(ErrorCode errorCode) NN_NOEXCEPT;

/**
* @brief        エラービューアを使用して EULA を表示します。
* @param[in]    region 表示する EULA のリージョン
*/
void ShowEula(nn::settings::system::RegionCode region) NN_NOEXCEPT;

/**
* @brief        エラービューアを使用して本体更新データの EULA を表示します。
* @param[in]    region 表示する EULA のリージョン
* @param[in]    eulaData 表示する EULA のデータ
* @pre
*               - eulaData が os::MemoryPageSize にアラインされている。
* @details      本体更新データの EULA を表示するための機能です。
*               region で指定したリージョンで必要となる可能性のある言語の EULA データを eulaData に設定した後呼び出してください。
*/
void ShowSystemUpdateEula(nn::settings::system::RegionCode region, EulaData& eulaData) NN_NOEXCEPT;

/**
* @brief        エラービューアを使用して記録されたエラーを表示します。
* @param[in]    result 記録されたエラーの Result 値
* @param[in]    timeOfOccurrence 協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数で表されたエラー発生時刻
* @details      本体設定のエラー履歴向けの機能です。
*/
void ShowErrorRecord(nn::Result result, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT;

/**
* @brief        エラービューアを使用して記録されたエラーを表示します。
* @param[in]    errorCode 記録されたエラーの ErrorRecord
* @param[in]    timeOfOccurrence 協定世界時 (UTC) の1970年1月1日午前0時0分0秒からの経過秒数で表されたエラー発生時刻
* @details      本体設定のエラー履歴向けの機能です。
*/
void ShowErrorRecord(ErrorCode errorCode, nn::time::PosixTime timeOfOccurrence) NN_NOEXCEPT;

/**
* @brief        エラービューアを使用して記録されたエラーを表示します。
* @param[in]    recordedErrorParam 起動パラメータ。
* @param[in]    recordedErrorParamSize 起動パラメータのサイズ。
* @pre
*               - recordedErrorParam と recordedErrorParamSize を CreateErrorViewerStartupParamForRecordedError 関数から取得
* @details      本体設定のエラー履歴向けの機能です。
*/
void ShowErrorRecord(const void* recordedErrorParam, size_t recordedErrorParamSize) NN_NOEXCEPT;

//! @}
//-----------------------------------------------------------------------------

}}
