﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief Error report manager class.
*/

#include <nn/os.h>
#include <nn/erpt/erpt_Types.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/sfdl/erpt.sfdl.h>

namespace nn   {
namespace erpt {

/**
 *  @brief Provides access to report summary and to the system event which is triggered when new report is created.
 */
class Manager
{
    NN_DISALLOW_COPY(Manager);
    NN_DISALLOW_MOVE(Manager);

private:
    nn::sf::SharedPointer<nn::erpt::sf::IManager> m_Impl;
    nn::os::SystemEvent m_SystemEvent;

public:
    /**
     *  @brief          Retrieve summary for all generated reports.
     *  @param[out]     reportList                              Pointer to @ref nn::erpt::ReportList storage.
     *  @details        A client may use this function to retrieve summary for all generated reports.
     *                  The summary will contain:
     *                  - number of reports
     *                  - id for each report
     *                  - user provided meta data (if any) for each report
     *                  - report flags
     *                  - timestamp
     *  @pre            Initialize must be called.
     */
    nn::Result GetReportList(ReportList& reportList)
    NN_NOEXCEPT;

    /**
     *  @brief          Retrieve filtered summary for all generated reports.
     *  @param[out]     reportList                              Pointer to @ref nn::erpt::ReportList storage.
     *  @param[in]      filterType                              Restricts summary to reports of type nn::erpt::ReportType.
     *  @retval         nn::erpt::ResultNotInitialized          Initialize has not been called.
     *  @details        A client may use this function to retrieve a summary for reports
     *                  with type specified by @a filterType parameter.
     *                  The summary will contain:
     *                  - number of reports
     *                  - id for each report
     *                  - user provided (if any) meta data for each report
     *                  - report flags
     *                  - timestamp
     *  @pre            Initialize must be called.
     */
    nn::Result GetReportList(ReportList& reportList, ReportType filterType)
    NN_NOEXCEPT;

    /**
     *  @brief          Obtain handle to system event which will get signaled whenever a new report is generated.
     *  @retval         Pointer to system event.
     *  @details        Clients wishing to receive notifications when new reports are generated may
     *                  choose to use this function to obtain a pointer to the system event. This event
     *                  will be signaled whenever a new report is generated.
     *  @pre            Initialize must be called.
     */
    nn::os::SystemEvent* GetEventPointer()
    NN_NOEXCEPT;

    /**
     *  @brief          Clean up all reports.
     *  @retval         nn::erpt::ResultNotInitialized          Initialize has not been called.
     *  @details        Clean up all reports stored in the system.
     *  @pre            Initialize must be called.
     */
    nn::Result CleanupReports()
    NN_NOEXCEPT;

    /**
     *  @brief          Delete a specified report.
     *  @retval         nn::erpt::ResultNotInitialized          Initialize has not been called.
     *  @retval         nn::erpt::InvalidArgument               The specified report is not found.
     *  @details        Delete an error report with specified ReportId.
     *  @pre            Initialize must be called.
     */
    nn::Result DeleteReport(ReportId reportId)
    NN_NOEXCEPT;

    /**
     *  @brief          Get storage usage statistics.
     *  @retval         nn::erpt::ResultNotInitialized          Initialize has not been called.
     *  @pre            Initialize must be called.
     */
    nn::Result GetStorageUsageStatistics(StorageUsageStatistics* pOut)
    NN_NOEXCEPT;

    /**
     *
     *  @brief          Initialize report manager.
     *  @retval         nn::erpt::ResultAlreadyInitialized      Initialize has already been called.
     *  @retval         nn::erpt::ResultOutOfMemory             Error service is out of memory.
     *  @details        This call will open a session to the error reporting service. In order to
     *                  utilize error report manager, clients should be provisioned for access to
     *                  "erpt:c" port.
     */
    nn::Result Initialize()
    NN_NOEXCEPT;

    /**
     *  @brief          Finalize report manager.
     *  @details        Closes session to the error reporting service.
     */
    void Finalize()
    NN_NOEXCEPT;

    Manager()
    NN_NOEXCEPT;

    ~Manager()
    NN_NOEXCEPT;
};

}}
